package SessionId;

use common::sense;

use CLASS;
use Moose;
use Moose::Util::TypeConstraints;
use MooseX::HasDefaults::RW;
use namespace::autoclean;
use Time::HiRes;

use Cookie;

our $POLICY_SESSIONAL = 'sessional';
our $POLICY_LONG      = 'long';
our $POLICY_MEDIUM    = 'medium';
our $POLICY_SHORT     = 'short';
our $POLICY_LONGER    = 'longer';

our $TYPE_NORMAL      = 'normal';
our $TYPE_LITE        = 'lite';

my %POLICY_NAME_TO_ID = (
    $POLICY_SESSIONAL => 0,
    $POLICY_LONG      => 1,
    $POLICY_MEDIUM    => 3,
    $POLICY_SHORT     => 4,
    $POLICY_LONGER    => 5,
);
my %POLICY_ID_TO_NAME  = reverse %POLICY_NAME_TO_ID;

my @AVAILABLE_POLICIES = map { $POLICY_ID_TO_NAME{$_} } sort keys %POLICY_ID_TO_NAME;
my @AVAILABLE_TYPES    = ($TYPE_NORMAL, $TYPE_LITE);

with 'MooseX::Clone';

enum 'SessionId.Policy' => \@AVAILABLE_POLICIES;
enum 'SessionId.Type'   => \@AVAILABLE_TYPES;

subtype 'PositiveInt'       => as 'Int'         => where { $_ > 0  };
subtype 'PositiveZeroInt'   => as 'Int'         => where { $_ >= 0 };
subtype 'Timestamp'         => as 'Int'         => where { $_ > 1_000_000_000 };
subtype 'FloatingTimestamp' => as 'Num'         => where { $_ == 0 or $_ > 1_000_000_000 };
subtype 'Passport.Uid'      => as 'PositiveInt';
subtype 'Passport.HostId'   => as 'Maybe[Str]'  => where { $_ =~ /^ (?: [0-9a-f]{2} )+ $/x };
subtype 'Passport.Cookie'   => as class_type('Cookie');

coerce 'Passport.HostId'       => from 'Str'     => via { lc $_ };
coerce 'Passport.Cookie'       => from 'HashRef' => via { Cookie->new(%$_) };

has 'created'             => (isa => 'Timestamp');

# Разница между значением времени на клиенте, которое приходит в POST-е при авторизации,
# и отметкой времени на сервере, на момент получения этого запроса. Измеряется в миллисекундах.
has 'delta'               => (isa => 'Int');
has 'uid'                 => (isa => 'PositiveInt');
has 'type'                => (isa => 'SessionId.Type',    is => 'ro', predicate => '_has_type',   writer => '_type');
has 'policy'              => (isa => 'SessionId.Policy',  is => 'ro', predicate => '_has_policy', writer => '_policy', default => $POLICY_SESSIONAL);

has 'tag'                 => (isa => 'Str');
has 'started'             => (isa => 'FloatingTimestamp');
has 'ip'                  => (isa => 'Str');
has 'passport_host'       => (isa => 'Str');
has 'is_betatester'       => (isa => 'Bool');
has 'is_yandexoid'        => (isa => 'Bool');
has 'social_profile_id'   => (isa => 'Maybe[PositiveInt]');
has 'is_stressed'         => (isa => 'Bool');

has 'user_have_password'   => (isa => 'Bool');
has 'password_verificated' => (isa => 'Maybe[Timestamp]');

has 'key_space_name'       => (isa => 'Str');

has 'cookie1'              => (isa => 'Passport.Cookie', default => sub { Cookie->new }, lazy => 1, coerce => 1, traits => ['Clone']);
has 'cookie2'              => (isa => 'Passport.Cookie', default => sub { Cookie->new }, lazy => 1, coerce => 1, traits => ['Clone']);

has 'string'               => (isa => 'Str');
has 'sslstring'            => (isa => 'Str');

has 'uids_validity'        => (isa => 'HashRef');
has 'uids_secure'          => (isa => 'HashRef');
has 'uids_number'          => (isa => 'PositiveInt');

my %FIELD_NAME_TO_VALUES = (
    policy => \@AVAILABLE_POLICIES,
    type   => \@AVAILABLE_TYPES,
);

{
    no strict 'refs';
    while (my ($field, $values) = each %FIELD_NAME_TO_VALUES) {
        for my $value (@$values) {
            my ($is_text, $be_text);

            $is_text .= qq/sub {/;
            $is_text .= qq/  die "$field isn't defined" unless \$_[0]->_has_$field;/;
            $is_text .= qq/  shift->$field eq '$value';/;
            $is_text .= qq/}/;

            $be_text  = qq/sub { shift->_$field('$value') }/;

            *{ "$CLASS:\:is_$value" } = eval $is_text;
            die $@ if $@;
            *{ "$CLASS:\:be_$value" } = eval $be_text;
            die $@ if $@;
        }
    }
}

sub age {
    my $self = shift;

    return time - $self->created unless @_;

    my $age = shift;
    $self->created(time - $age);

    return $self;
}

sub policy_id {
    my $self = shift;

    return $POLICY_NAME_TO_ID{ $self->policy } unless @_;

    my $policy_id = shift;
    my $value = $POLICY_ID_TO_NAME{$policy_id};
    $self->_policy($value);

    return $self;
}

sub password_verification_age {
    my $self = shift;

    unless (@_) {
        return undef unless $self->password_verificated;
        return time - $self->password_verificated;
    }

    my $age = shift;
    my $value
      = $age == -1
      ? undef
      : time - $age;
    $self->password_verificated($value);

    return $self;
}

sub update_password_verificated { shift->password_verificated(time) }

__PACKAGE__->meta->make_immutable;

1;
