#!/bin/bash

#set -e
#set -x


# passport prod settings
OCSP_RESPONDER="http://ocsp2.globalsign.com/gsextendvalsha2g2"
CERTS_DIR="/etc/nginx/certs"
CERTIFICATE_FILE="$CERTS_DIR/passport.yandex.ru.crt"
INTERMEDIATE_FILE="$CERTS_DIR/gsextendvalsha2g2r2.pem"
INTERMEDIATE_AND_ROOT_FILE="$CERTS_DIR/gsextendvalsha2g2r2_and_gssha1rootr2.pem"
OCSP_DIR="$CERTS_DIR/ocsp-gs"
OCSP_STAPLE_LINK="$CERTS_DIR/ocsp-gs.der"
TMP_PREFIX="/tmp/ocsp-gs"
LOCK_FILE="/tmp/process-ocsp-stapling-gs.lock"
OCSP_MINUTES_MIN_TTL=235 # 3 hours 55 minutes
OCSP_MINUTES_MAX_TTL=10080 # 1 week
CURL_TIMEOUT=15
CURL_RETRIES=2
CURL_DELAY=15
SUPPORTED_SERVICE_ENV="production stress"

# passport dev settings
#OCSP_RESPONDER="http://ocsp2.globalsign.com/gsextendvalsha2g2"
#CERTS_DIR="/home/spleenjack/crt"
#CERTIFICATE_FILE="$CERTS_DIR/passport.yandex.ru.201511.crt"
#INTERMEDIATE_FILE="$CERTS_DIR/gsextendvalsha2g2r2.pem"
#INTERMEDIATE_AND_ROOT_FILE="$CERTS_DIR/gsextendvalsha2g2r2_and_gssha1rootr2.pem"
#OCSP_DIR="$CERTS_DIR/ocsp-gs"
#OCSP_STAPLE_LINK="$CERTS_DIR/ocsp-gs.der"
#TMP_PREFIX="/tmp/ocsp-gs"
#LOCK_FILE="/tmp/process-ocsp-stapling-gs.lock"
#OCSP_MINUTES_MIN_TTL=235 # 3 hours 55 minutes
#OCSP_MINUTES_MAX_TTL=10080 # 1 week
#CURL_TIMEOUT=15
#CURL_RETRIES=2
#CURL_DELAY=15
#SUPPORTED_SERVICE_ENV="production stress"


# mda prod settings
#OCSP_RESPONDER="http://ocsp2.globalsign.com/gsorganizationvalsha2g2"
#CERTS_DIR="/etc/nginx/certs"
#CERTIFICATE_FILE="$CERTS_DIR/pass.yandex.ru-gs-sha2.crt"
#INTERMEDIATE_FILE="$CERTS_DIR/gsorganizationvalsha2g2r1.pem"
#INTERMEDIATE_AND_ROOT_FILE="$CERTS_DIR/gsorganizationvalsha2g2r1_and_gssha1rootr1.pem"
#OCSP_DIR="$CERTS_DIR/ocsp-mda-gs"
#OCSP_STAPLE_LINK="$CERTS_DIR/ocsp-mda-gs.der"
#TMP_PREFIX="/tmp/ocsp-mda-gs"
#LOCK_FILE="/tmp/process-ocsp-stapling-mda-gs.lock"
#OCSP_MINUTES_MIN_TTL=235 # 3 hours 55 minutes
#OCSP_MINUTES_MAX_TTL=10080 # 1 week
#CURL_TIMEOUT=15
#CURL_RETRIES=2
#CURL_DELAY=15
#SUPPORTED_SERVICE_ENV="production stress"

# mda dev settings
#OCSP_RESPONDER="http://ocsp2.globalsign.com/gsorganizationvalsha2g2"
#CERTS_DIR="/home/spleenjack/crt"
#CERTIFICATE_FILE="$CERTS_DIR/pass.yandex.ru-gs-sha2.crt"
#INTERMEDIATE_FILE="$CERTS_DIR/gsorganizationvalsha2g2r1.pem"
#INTERMEDIATE_AND_ROOT_FILE="$CERTS_DIR/gsorganizationvalsha2g2r1_and_gssha1rootr1.pem"
#OCSP_DIR="$CERTS_DIR/ocsp-mda-gs"
#OCSP_STAPLE_LINK="$CERTS_DIR/ocsp-mda-gs.der"
#TMP_PREFIX="/tmp/ocsp-mda-gs"
#LOCK_FILE="/tmp/process-ocsp-stapling-mda-gs.lock"
#OCSP_MINUTES_MIN_TTL=5
#OCSP_MINUTES_MAX_TTL=100
#CURL_TIMEOUT=5
#CURL_RETRIES=2
#CURL_DELAY=15
#SUPPORTED_SERVICE_ENV="development testing"



check_environment () {
    SERVICE_ENV=`cat /etc/yandex/environment.type`
    if [[ $SUPPORTED_SERVICE_ENV != *$SERVICE_ENV* ]]
    then
        echo "unsupported environment: $SERVICE_ENV"
        exit 0
    fi
}


lock () {
    exec 200>$LOCK_FILE

    flock -n 200 \
        && return 0 \
        || return 1
}


create_ocsp_dir () {
    [ -d $OCSP_DIR ] || mkdir -p $OCSP_DIR
}


get_new_ocsp_response () {
    CURRENT_TIME=`date '+%Y-%m-%d.%H-%M-%S'`
    OCSP_REQUEST_TMP_FILE="$TMP_PREFIX.$CURRENT_TIME.req"
    OCSP_RESPONSE_TMP_FILE="$TMP_PREFIX.$CURRENT_TIME.der"
    OCSP_RESPONSE_NEW_FILE="$OCSP_DIR/$CURRENT_TIME.der"

    echo "generate ocsp request: $OCSP_REQUEST_TMP_FILE"
    OPENSSL_STDOUT=`
        openssl ocsp \
          -issuer $INTERMEDIATE_FILE \
          -cert   $CERTIFICATE_FILE \
          -reqout $OCSP_REQUEST_TMP_FILE \
          -no_nonce \
          2>&1
    `

    if [ $? -eq 0 ] && [ -e "$OCSP_REQUEST_TMP_FILE" ]
    then
        OCSP_REQUEST_B64=`base64 -w 0 < $OCSP_REQUEST_TMP_FILE`

        echo "download tmp ocsp response: $OCSP_RESPONSE_TMP_FILE"
        CURL_STDOUT=`
            curl \
              --silent \
              --show-error \
              --url        "$OCSP_RESPONDER/$OCSP_REQUEST_B64" \
              --output      $OCSP_RESPONSE_TMP_FILE \
              --max-time    $CURL_TIMEOUT \
              --retry       $CURL_RETRIES \
              --retry-delay $CURL_DELAY \
              2>&1
        `

        if [ $? -eq 0 ] && [ -e "$OCSP_RESPONSE_TMP_FILE" ]
        then
            echo "verify tmp ocsp response"
            OPENSSL_STDOUT=`
                openssl ocsp \
                  -CAfile  $INTERMEDIATE_AND_ROOT_FILE \
                  -issuer  $INTERMEDIATE_FILE \
                  -cert    $CERTIFICATE_FILE \
                  -respin  $OCSP_RESPONSE_TMP_FILE \
                  -no_nonce \
                  2>&1
            `

            if [ $? -eq 0 ] && [[ $OPENSSL_STDOUT == *"Response verify OK"* ]]
            then
                echo "$OPENSSL_STDOUT"
                echo "move tmp to new ocsp response: $OCSP_RESPONSE_NEW_FILE"
                mv $OCSP_RESPONSE_TMP_FILE $OCSP_RESPONSE_NEW_FILE
            else
                echo "failed verifying tmp ocsp response"
                echo "$OPENSSL_STDOUT"
                >&2 echo "$OPENSSL_STDOUT"
            fi

        else
            echo "failed downloading ocsp response"
            echo "$CURL_STDOUT"
            >&2 echo "$CURL_STDOUT"
        fi

    else
        echo "failed generating ocsp request"
        echo "$OPENSSL_STDOUT"
        >&2 echo "$OPENSSL_STDOUT"
    fi
}


choose_ocsp_staple () {
    echo "choose first ocsp staple older than min ttl: $OCSP_MINUTES_MIN_TTL"
    OCSP_STAPLE_CHOOSEN_FILE=`find $OCSP_DIR -mmin +$OCSP_MINUTES_MIN_TTL -type f -printf '%T@\t%p\n' | sort -nr | head -1 | cut -f2`

    if [ -z "$OCSP_STAPLE_CHOOSEN_FILE" ]
    then
        echo "choose first ocsp staple younger than min ttl"
        OCSP_STAPLE_CHOOSEN_FILE=`find $OCSP_DIR -mmin -$OCSP_MINUTES_MIN_TTL -type f -printf '%T@\t%p\n' | sort -n | head -1 | cut -f2`
    fi
}


create_ocsp_staple_link () {
    if [ ! -z "$OCSP_STAPLE_CHOOSEN_FILE" ]
    then
        echo "choosen ocsp staple: $OCSP_STAPLE_CHOOSEN_FILE"
        echo "create ocsp staple symlink: $OCSP_STAPLE_LINK"
        ln -sf $OCSP_STAPLE_CHOOSEN_FILE $OCSP_STAPLE_LINK
    fi
}


clean_old_ocsp_responses () {
    echo "clean responses older than $OCSP_MINUTES_MAX_TTL minutes"
    OCSP_STAPLE_CURRENT_NAME=$(basename `readlink -f $OCSP_STAPLE_LINK`)
    find $OCSP_DIR -type f -mmin +$OCSP_MINUTES_MAX_TTL -not -name $OCSP_STAPLE_CURRENT_NAME -delete
}


reload_nginx () {
    /etc/init.d/nginx reload
}


main () {
    lock || exit 1
    check_environment
    create_ocsp_dir
    get_new_ocsp_response
    choose_ocsp_staple
    create_ocsp_staple_link
    clean_old_ocsp_responses
    reload_nginx
    echo
    echo
}


main
