#!/usr/bin/env perl

use common::sense;

use Data::Dumper;
use Digest::SHA qw/hmac_sha256 hmac_sha256_hex sha256/;
use Getopt::Long;
use URI::Escape;
use MIME::Base64 qw/decode_base64url decode_base64 encode_base64url/;
use MIME::Base32 'RFC';

my $INTERVAL = 30;
my $SIZEOF_INT64 = 8;
my $DEBUG = 1;

sub str2hex ($)  { unpack 'H*', shift }
sub hex2str ($)  {   pack 'H*', shift }
sub str2b64 ($)  { encode_base64url shift, '' }
sub b642str ($)  { decode_base64url shift     }
sub b322str ($)  { MIME::Base32::decode(shift) }
sub str2b32 ($)  { MIME::Base32::encode(shift) }
sub str2int ($)  { int shift }
sub hex2bigint ($)  { use bigint; hex shift }
sub bigint2hex ($)  { use bigint; substr shift->as_hex, 2 }

sub ts2utc ($) {
    my ($sec, $min, $hour, $day, $mon, $year) = gmtime shift;
    return sprintf '%04d-%02d-%02dT%02d:%02d:%02d', $year + 1900, $mon + 1, $day, $hour, $min, $sec;
}



sub truncate_hash ($) {
=pod
std::string MakeLettersOtp (const std::string& hs)
{
    const std::size_t offset = hs[hs.length() - 1] & 0xf;

    int64_t binary = int64_t(hs[offset] & 0x7f) << (sizeof(int64_t) - 1) * 8;
    for (std::size_t i = 1; i != sizeof(int64_t); ++i)
        binary |= int64_t(hs[offset + i] & 0xff) << (sizeof(int64_t) - i - 1) * 8;
       
    binary %= 208827064576L; // 26 ^ 8

    std::string s(8, 'a');
    for (std::string::reverse_iterator p = s.rbegin(); p != s.rend(); ++p) {
        *p = static_cast<char>(binary % 26) + 'a';
        binary /= 26;
    }
    
    return s;
=cut
    my $bytes = shift;

    printf "bytes = 0x%s\n", str2hex $bytes if $DEBUG;

    my @bytes  = split //, $bytes;
    my $offset =  ord $bytes[-1]      & 0x0f;
    my $binary = (ord $bytes[$offset] & 0x7f) << ($SIZEOF_INT64 - 1) * 8;

    printf "offset = bytes[-1] & 0x0f = 0x%s & 0x0f = %i\n",
      str2hex $bytes[-1],
      $offset
      if $DEBUG;

    printf "binary  = (bytes[offset    ] & 0x7f) << (sizeof(int64)     - 1) * 8 = (0x%s & 0x7f) << %2i = 0x%s << %2i = 0x%x\n",
      str2hex $bytes[$offset],
      ($SIZEOF_INT64 - 1) * 8,
      str2hex chr(ord $bytes[$offset] & 0x7f),
      ($SIZEOF_INT64 - 1) * 8,
      $binary
      if $DEBUG;

    for my $i (1 .. $SIZEOF_INT64 - 1) {
        $binary |= (ord $bytes[$offset + $i] & 0xff) << ($SIZEOF_INT64 - $i - 1) * 8;

        printf "binary |= (bytes[offset + i] & 0xff) << (sizeof(int64) - i - 1) * 8 = (0x%s & 0xff) << %2i = 0x%s << %2i = 0x%x\n",
          str2hex $bytes[$offset + $i],
          ($SIZEOF_INT64 - $i - 1) * 8,
          str2hex chr(ord $bytes[$offset + $i] & 0xff),
          ($SIZEOF_INT64 - $i - 1) * 8,
          $binary
          if $DEBUG;
    }

    $binary %= 208827064576; # 26 ^ 8

    printf "binary %%= 26^8 = 0x%x = %i\n", $binary, $binary if $DEBUG;

    my $result;

    for (1 .. 8) {
        $result = chr(ord('a') + $binary % 26) . $result;
        $binary /= 26;
    }

    return $result;
}

my %opt;

GetOptions(
    \%opt,
    '1',
    'secret|s=s',
    'pin|p=s',
#    'secretpin|sp=s',
    'from|f=i',
    'to|t=i',
    'at=i',
    'skew=i',
    'verbose|v',
);

$DEBUG = $opt{verbose};

#my $secretpin_str       = decode_base64 uri_unescape $opt{secret};
my $secret_str       = b322str $opt{secret};
my $pin_str          = $opt{pin};

my $secret_hex = str2hex $secret_str;
my $secret_int = hex2bigint($secret_hex);
my $pin_int    = str2int $pin_str;

my $secretpin_int = $secret_int * 10_000 + $pin_int;
my $secretpin_hex = bigint2hex $secretpin_int;
my $secretpin_str = hex2str $secretpin_hex;

my $first_window = int($opt{from} / $INTERVAL);
my $last_window  = int($opt{to}   / $INTERVAL);

if ($opt{at}) {
    $first_window = int( ($opt{at} - $opt{skew}) / $INTERVAL );
    $last_window  = int( ($opt{at} + $opt{skew}) / $INTERVAL );
}

my $key_str
  = $opt{1}
  ? $secretpin_str
  : sha256($pin_str . $secret_str)
  ;
my $key_hex = str2hex $key_str;

printf "secret = %s = 0x%s\n", "$secret_int", $secret_hex if $DEBUG;
printf "pin = %s\n", $pin_str if $DEBUG;
if ($opt{1}) {
    printf "secretpin = secret * 10000 + pin = %s = 0x%s\n\n", $secretpin_int, $secretpin_hex if $DEBUG;
}
else {
    printf "key = sha256(pin || secret) = 0x%s\n\n", $key_hex if $DEBUG;
}
printf "first_window = %s\n", ts2utc $first_window * $INTERVAL if $DEBUG;
printf "last_window = %s\n", ts2utc $last_window * $INTERVAL if $DEBUG;

for my $window ($first_window .. $last_window) {
    my $data = pack "q>", $window;

    printf "\ndata = window = 0x%s\n", str2hex $data if $DEBUG;

    my $hash = hmac_sha256 $data, $key_str;
    my $totp = truncate_hash $hash;

    my $window_open_time  =  $window      * $INTERVAL;
    my $window_close_time = ($window + 1) * $INTERVAL - 1;

    printf "%s\topened=%d\tclosed=%d\twindow=%d\topenedutc=%s\n", $totp, $window_open_time, $window_close_time, $window, ts2utc $window_open_time;
}
