# -*- coding: utf-8 -*-
import logging
from logging.config import dictConfig
import sys

from flask import (
    Flask,
    jsonify,
    redirect,
    request,
)
from passport.backend.pocket_service.utils.helpers import string_to_bool
from passport.backend.pocket_service.utils.secrets import get_secrets
from passport.backend.pocket_service.utils.tvm import init_tvm_client
from passport.backend.pocket_service.views import (
    current_release,
    deployment_ticket,
    rebase,
)
from passport.backend.pocket_service.webhooks import (
    conductor_webhooks,
    github_webhooks,
    other_webhooks,
    startrek_webhooks,
)


app = Flask(__name__)


def setup_logging():
    dictConfig({
        'version': 1,
        'disable_existing_loggers': False,
        'formatters': {
            'default': {
                'format': '[%(asctime)s] %(levelname)s %(message)s',
            },
        },
        'handlers': {
            'console': {
                'class': 'logging.StreamHandler',
                'stream': sys.stdout,
                'formatter': 'default',
            },
        },
        'root': {
            'level': 'DEBUG',
            'handlers': ['console'],
        },
    })


setup_logging()


@app.before_request
def log_request():
    logging.debug('Got request from %s: %s %s', request.remote_addr, request.method, request.full_path.rstrip('?'))


@app.after_request
def log_response(response):
    logging.debug('Sent response with status %s', response.status)
    return response


@app.route('/', methods=['GET', 'POST'])
def hello_view():
    return 'Hello from Flask'


@app.route('/ping', methods=['GET'])
def ping():
    return 'Pong'


@app.route('/webhooks/conductor/ready_for_test', methods=['POST'])
def conductor_ready_for_test_view():
    try:
        request_data = request.get_json(force=True)
    except Exception as e:
        logging.error('{}: {}'.format(e, request.data))
        raise
    transitioned_tickets = conductor_webhooks.conductor_ready_for_test_controller(request_data)
    return jsonify(
        {
            'status': 'ok',
            'transitioned_tickets': transitioned_tickets,
        },
    )


@app.route('/webhooks/conductor/close', methods=['POST'])
def conductor_close_view():
    try:
        request_data = request.get_json(force=True)
    except Exception as e:
        logging.error('{}: {}'.format(e, request.data))
        raise
    transitioned_tickets = conductor_webhooks.conductor_close_controller(request_data, resolution='fixed')
    return jsonify(
        {
            'status': 'ok',
            'transitioned_tickets': transitioned_tickets,
        },
    )


@app.route('/webhooks/conductor/run_molly_scan', methods=['POST'])
def conductor_run_molly_scan_view():
    try:
        request_data = request.get_json(force=True)
    except Exception as e:
        logging.error(str(e) + ': ' + str(request.data))
        raise
    scan_ids = conductor_webhooks.conductor_molly_controller(request_data)
    return jsonify(
        {
            'status': 'ok',
            'scan_id': scan_ids,
        },
    )


@app.route('/webhooks/conductor/add_changelog_comment_to_release_ticket', methods=['POST'])
def conductor_add_changelog_comment_to_release_ticket():
    try:
        request_data = request.get_json(force=True)
    except Exception as e:
        logging.error('{}: {}'.format(e, request.data))
        raise
    release_ticket, linked_tickets = conductor_webhooks.conductor_add_changelog_comment_controller(request_data)
    return jsonify(
        {
            'status': 'ok',
            'release_ticket': release_ticket,
            'linked_tickets': linked_tickets,
        },
    )


@app.route('/webhooks/deploying_started', methods=['POST'])
def deploying_started_view():
    try:
        request_data = request.get_json(force=True)
    except Exception as e:
        logging.error('{}: {}'.format(e, request.data))
        raise
    other_webhooks.package_deploying_started(request_data)
    return jsonify(
        {
            'status': 'ok',
        },
    )


@app.route('/webhooks/conductor/deploying_finished/<service_id>/<environment_id>/', methods=['POST'])
def conductor_deploying_finished_view(service_id, environment_id):
    try:
        request_data = request.get_json(force=True)
    except Exception as e:
        logging.error('{}: {}'.format(e, request.data))
        raise
    conductor_webhooks.conductor_package_deploying_finished(
        request_data,
        service_id=service_id,
        environment_id=environment_id,
    )
    return jsonify(
        {
            'status': 'ok',
        },
    )


@app.route('/webhooks/github/resolve_on_merge', methods=['POST'])
def github_resolve_on_merge():
    try:
        request_data = request.get_json(force=True)
    except Exception as e:
        logging.error('{}: {}'.format(e, request.data))
        raise
    transitioned_tickets = github_webhooks.github_resolve_on_pr_merge(request_data)
    return jsonify(
        {
            'status': 'ok',
            'transitioned_tickets': transitioned_tickets,
        },
    )


@app.route('/webhooks/github/add_links_on_pr_open', methods=['POST'])
def github_add_links_on_pr_open():
    try:
        request_data = request.get_json(force=True)
    except Exception as e:
        logging.error('{}: {}'.format(e, request.data))
        raise

    additional_params = {}
    for param in ['include_tickets', 'include_teamcity', 'include_rebase']:
        try:
            value = request.args.get(param)
            if value:
                value = value[0]
                additional_params[param] = string_to_bool(value)
        except ValueError:
            pass

    data = github_webhooks.github_add_links_on_pr_open(request_data, **additional_params)
    return jsonify(
        {
            'status': 'ok',
            'data': data,
        },
    )


@app.route('/webhooks/startrek/assign_to_duty_person', methods=['POST'])
def assign_to_duty_person_view():
    try:
        request_data = request.get_json(force=True)
    except Exception as e:
        logging.error('{}: {}'.format(e, request.data))
        raise

    startrek_webhooks.assign_to_duty_person(request_data)
    return jsonify(
        {
            'status': 'ok',
        },
    )


@app.route('/webhooks/startrek/set_current_sprint', methods=['POST'])
def set_current_sprint_view():
    try:
        request_data = request.get_json(force=True)
    except Exception as e:
        logging.error('{}: {}'.format(e, request.data))
        raise

    startrek_webhooks.set_current_sprint(request_data)
    return jsonify(
        {
            'status': 'ok',
        },
    )


@app.route('/diag/request', methods=['GET', 'POST'])
def request_view():
    return jsonify(
        {
            'args': request.args.items(),
            'form': request.form.items(),
            'cookies': request.cookies.items(),
            'headers': request.headers.items(),
            'data': request.data,
            'method': request.method,
            'path': request.path,
        }
    )


@app.route('/diag/secrets')
def secrets_view():
    secrets = get_secrets()
    return jsonify(
        {
            'status': 'ok',
            'secrets': list(secrets),
        },
    )


@app.route('/passport/current_release')
def passport_current_release():
    ticket = current_release.get_passport_current_release()
    return redirect(ticket, code=302)


@app.route('/passport/rebase/<owner>/<project>/<int:pr_no>', methods=['GET'])
def passport_rebase(owner, project, pr_no):
    result = rebase.do_rebase(owner, project, pr_no)
    response = jsonify(
        {
            'status': 'ok',
            'teamcity_response': result,
        },
    )
    response.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    response.headers['Pragma'] = 'no-cache'
    response.headers['Expires'] = '0'
    return response


@app.route('/frontend/current_release')
def frontend_current_release():
    ticket = current_release.get_frontend_current_release()
    return redirect(ticket, code=302)


@app.route('/am/synchronize_release_versions_with_tickets')
def am_synchronize_release_versions_with_tickets():
    current_release.synchronize_am_release_ticket_and_version()
    return jsonify(
        {
            'status': 'ok',
        },
    )


@app.route('/deployment_ticket/from_conductor_ticket/<int:conductor_ticket_id>', methods=['POST'])
def create_deployment_ticket(conductor_ticket_id):
    ticket = deployment_ticket.create_deployment_ticket_from_conductor_ticket(
        conductor_ticket_id=conductor_ticket_id,
        st_queue=request.form.get('queue', 'PASSPADMIN'),
        assignee=request.form.get('assignee'),
        session_id=request.cookies.get('Session_id'),
    )
    return redirect(ticket, code=302)


def post_worker_init(_worker):
    init_tvm_client()


def execute_app():
    return app


if __name__ == '__main__':
    app.run(debug=False)
