from concurrent import futures
import os

import requests
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry


def get_verify():
    if os.environ.get('SSL_DONT_VERIFY'):
        return False
    return os.environ.get('YA_INTERNAL_ROOT_CA')


def requests_retry_session(
        retries=3,
        backoff_factor=0.3,
        method_whitelist=('GET', 'POST'),
        status_forcelist=(500, 502, 504, 503),
        session=None,
):
    session = session or requests.Session()
    retry = Retry(
        total=retries,
        read=retries,
        connect=retries,
        backoff_factor=backoff_factor,
        method_whitelist=method_whitelist,
        status_forcelist=status_forcelist,
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


def execute_one(request):
    response = requests_retry_session().send(request, verify=get_verify())
    return response


def execute_many(requests, max_workers=5, **kwargs):
    request_to_idx = {request: i for i, request in enumerate(requests)}
    retval = [None] * len(requests)
    verify = get_verify()
    with futures.ThreadPoolExecutor(max_workers=max_workers) as executor:
        future_to_url = dict((executor.submit(requests_retry_session(**kwargs).send, request, verify=verify), request)
                             for request in requests)

        for future in futures.as_completed(future_to_url):
            request = future_to_url[future]
            i = request_to_idx[request]
            if future.exception() is not None:
                retval[i] = (False, future.exception())
            else:
                retval[i] = (True, future.result())
    return retval
