import json
from time import time
from urllib.parse import urljoin

from passport.backend.pocket_service.utils.secrets import get_secrets
import requests


_API_URL = 'https://infra-api.yandex-team.ru'


class InfraInvalidResponse(Exception):
    pass


class Infra(object):
    """
    https://infra-api.yandex-team.ru/swagger/
    """
    def __init__(self, url=_API_URL, token=None):
        self.url = url
        self.token = token

    def oauth_header(self):
        return {'Authorization': 'OAuth {token}'.format(token=self.token)}

    def create_event(self, service_id, environment_id, meta=None, **kwargs):
        meta = dict(
            event_type='package_deploying',
            **(meta or dict())
        )
        start_time = int(time())
        event_data = dict(
            environmentId=environment_id,
            serviceId=service_id,
            startTime=start_time,
            type='maintenance',
            severity='minor',
            setAllAvailableDc=True,
            meta=meta,
            **kwargs
        )
        response = requests.post(
            url=urljoin(self.url, 'v1/events/'),
            headers=self.oauth_header(),
            json=event_data,
        ).json()
        if 'id' not in response:
            raise InfraInvalidResponse(
                'Invalid response from infra: {}'.format(
                    json.dumps(
                        response,
                        indent=2,
                        sort_keys=True,
                    )),
            )
        return response['id']

    def update_event(self, event_id, **kwargs):
        requests.put(
            url=urljoin(self.url, 'v1/events/{}'.format(event_id)),
            headers=self.oauth_header(),
            json=kwargs,
        )

    def delete_event(self, event_id):
        requests.delete(
            url=urljoin(self.url, 'v1/events/{}'.format(event_id)),
            headers=self.oauth_header(),
        )

    def get_events(self, service_id, environment_id, delta=60*60*24, only_ongoing=True):
        """
        Находит события, у которых дата завершения позже чем последние delta секунд
        """
        now = int(time())
        _from = now - delta
        import logging
        logging.debug(_from)
        url = 'v1/events/current/' if only_ongoing else 'v1/events/'
        events = requests.get(
            url=urljoin(self.url, url),
            params={
                'from': _from,
                'environmentId': environment_id,
                'serviceId': service_id,
            },
            headers=self.oauth_header(),
        ).json()
        events = [event for event in events if event.get('meta', {}).get('event_type') == 'package_deploying']
        return events


def get_infra():
    return Infra(token=get_secrets()['INFRA_SECRET'])
