import json
from urllib.parse import urljoin

from passport.backend.pocket_service.utils.secrets import get_secrets
import requests
from startrek_client import Startrek


class StarTrekException(Exception):
    pass


class StarTrek(object):
    """
    https://wiki.yandex-team.ru/tracker/api/
    """
    def __init__(self, url='https://st-api.yandex-team.ru/', token=None, session_id=None):
        self.url = url
        self.token = token
        self.session_id = session_id

    def auth_header(self):
        # Если можем - ходим от имени пользователя. Иначе - от имени робота.
        if self.session_id is not None:
            return {'Cookie': 'Session_id=%s' % self.session_id}
        else:
            return {'Authorization': 'OAuth %s' % self.token}

    def get_transitions(self, ticket):
        return requests.Request(
            method='GET',
            url=urljoin(self.url, 'v2/issues/{ticket}/transitions'.format(ticket=ticket)),
            headers=self.auth_header(),
        ).prepare()

    def do_transition(self, ticket, status, resolution=None):
        if resolution is not None:
            data = json.dumps({'resolution': resolution})
        else:
            data = None
        return requests.Request(
            method='POST',
            url=urljoin(self.url, 'v2/issues/{ticket}/transitions/{status}/_execute'.format(
                ticket=ticket,
                status=status.lower(),
            )),
            headers=self.auth_header(),
            data=data,
        ).prepare()

    def search(self, query, page=1, per_page=10):
        return requests.Request(
            method='GET',
            url=urljoin(self.url, 'v2/issues'),
            params={
                'query': query,
                'page': page,
                'perPage': per_page,
            },
            headers=self.auth_header(),
        ).prepare()

    def create_ticket(self, queue, title, body, assignee=None):
        data = {
            'queue': queue,
            'summary': title,
            'description': body,
        }
        if assignee is not None:
            data.update(assignee=assignee)

        return requests.Request(
            method='POST',
            url=urljoin(self.url, 'v2/issues'),
            json=data,
            headers=self.auth_header(),
        ).prepare()


def get_startrek(session_id=None):
    return StarTrek(token=get_secrets()['STARTREK_SECRET'], session_id=session_id)


def get_startrek_client(session_id=None):
    return Startrek(
        useragent='passport-pocket-service/python',
        token=get_secrets()['STARTREK_SECRET'],
        session_id=session_id,
    )
