import logging
import re

from passport.backend.pocket_service.utils.http import execute_many
from passport.backend.pocket_service.utils.startrek import (
    get_startrek,
    get_startrek_client,
)


RE_TICKET = re.compile(r'[A-Z]{2,}-\d+')
RE_PASSP = re.compile(r'PASSP-\d{5}')
RE_VAULT = re.compile(r'VAULT-\d+')
RE_PASSP_RELEASE = re.compile(r'https://st\.yandex-team\.ru/PASSP-\d{5}')


def find_tickets(text):
    tickets = RE_PASSP.findall(text) + RE_VAULT.findall(text)
    return tickets


def find_first_ticket(text):
    ticket_match = RE_TICKET.search(text)
    if ticket_match:
        return ticket_match.group()
    return None


def find_release_ticket(text):
    """Считаем, что первая указанная ссылка на тикет в очереди PASSP — это ссылка на релизный тикет"""
    ticket_link = RE_PASSP_RELEASE.search(text)
    if ticket_link:
        ticket = RE_PASSP.search(ticket_link.group()).group()
        return ticket
    return None


def url_to_ticket(ticket):
    return 'https://st.yandex-team.ru/{ticket}'.format(ticket=ticket)


def find_tickets_for_transition(tickets, status):
    retval = []

    st = get_startrek()
    requests = []
    for ticket in tickets:
        requests.append(st.get_transitions(ticket))
    responses = execute_many(requests)

    for i, (is_ok, response) in enumerate(responses):
        if not is_ok:
            continue
        possible_transitions = response.json()
        for transition in possible_transitions:
            if 'id' not in transition:
                continue
            if transition['id'] == status:
                retval.append(tickets[i])
                break
        else:
            logging.info("Ticket {ticket}: no transition to {status}".format(status=status, ticket=tickets[i]))
    return retval


def do_transition(tickets, status, resolution=None):
    if not tickets:
        return []

    retval = []

    st = get_startrek()
    requests = []
    for ticket in tickets:
        requests.append(st.do_transition(ticket, status, resolution=resolution))

    responses = execute_many(requests)
    # FIXME сейчас 404 игнорируется и не считается ошибкой
    for i, (is_ok, response) in enumerate(responses):
        if not is_ok:
            continue
        retval.append(tickets[i])
    return retval


def attach_molly_report_links(release_ticket, scan_ids):
    report_links = ['* https://molly.yandex-team.ru/report/{}?progressive=True'.format(scan_id) for scan_id in scan_ids]
    molly_reports_prefix = u'Тестирование в Молли:'

    comment_with_molly_reports = None
    comments = get_startrek_client().issues[release_ticket].comments
    for comment in comments:
        if comment['text'].startswith(molly_reports_prefix):
            comment_with_molly_reports = comment
            break
    if not comment_with_molly_reports:
        message = ' \n'.join([molly_reports_prefix] + report_links)
        comments.create(text=message, params={'notify': False, 'isAddToFollowers': False})
    else:
        message = ' \n'.join([comment_with_molly_reports['text']] + report_links)
        comment_with_molly_reports.update(text=message, params={'notify': False, 'isAddToFollowers': False})
