import logging
from time import time

from passport.backend.pocket_service.utils.http import execute_one
from passport.backend.pocket_service.utils.infra import get_infra
from passport.backend.pocket_service.utils.molly import (
    get_molly,
    get_molly_params,
)
from passport.backend.pocket_service.utils.startrek import get_startrek_client
from passport.backend.pocket_service.webhooks.common import (
    attach_molly_report_links,
    do_transition,
    find_first_ticket,
    find_release_ticket,
    find_tickets,
    find_tickets_for_transition,
)


def is_final_call(conductor_request):
    if 'ticket_tasks_total' in conductor_request and 'ticket_tasks_finished' in conductor_request:
        finished = conductor_request['ticket_tasks_finished']
        total = conductor_request['ticket_tasks_total']
        return finished == total
    else:
        return True


def conductor_ready_for_test_controller(conductor_request):
    if not is_final_call(conductor_request):
        return
    tickets = find_tickets(conductor_request['comment'])
    tickets_for_transition = find_tickets_for_transition(tickets, 'ready_for_test')
    return do_transition(tickets_for_transition, 'ready_for_test')


def conductor_close_controller(conductor_request, resolution):
    if not is_final_call(conductor_request):
        return
    tickets = find_tickets(conductor_request['comment'])
    tickets_for_transition = find_tickets_for_transition(tickets, 'close')
    return do_transition(tickets_for_transition, 'close', resolution)


def conductor_add_changelog_comment_controller(conductor_request):
    conductor_comment = conductor_request['comment']
    conductor_comment_lines = [line for line in conductor_comment.splitlines() if line.strip()]
    release_ticket = find_release_ticket(conductor_comment_lines[0])
    if release_ticket:
        escaped_conductor_comment = '\n'.join(['%%'] + conductor_comment_lines[1:] + ['%%'])
        st = get_startrek_client()
        issue = st.issues[release_ticket]
        if issue.description != escaped_conductor_comment:
            has_comment_with_changelog = False
            for comment in issue.comments:
                if comment['text'] == escaped_conductor_comment:
                    has_comment_with_changelog = True
                    break
            if not has_comment_with_changelog:
                issue.comments.create(
                    text=escaped_conductor_comment,
                    params={'notify': False, 'isAddToFollowers': False},
                )
                linked_to_release = {i.key for i in st.issues.find('"Linked To": {}'.format(release_ticket))}
                tickets = {ticket for ticket in [find_first_ticket(li) for li in conductor_comment_lines[1:]] if ticket}
                tickets_to_link = list(tickets - linked_to_release)
                st.bulkchange.update(
                    issues=tickets_to_link,
                    links=[{'relationship': 'relates', 'issue': release_ticket}],
                    followers={'add': []},  # нельзя изменить только связи в тикете, поэтому как бы меняем наблюдателей
                ).wait()
                return release_ticket, tickets_to_link
    return release_ticket, []


def conductor_molly_controller(conductor_request):
    if not is_final_call(conductor_request):
        return
    target_package_names = [pkg_description['package'] for pkg_description in conductor_request['packages']]
    target_env = conductor_request['branch']
    scan_ids = []
    for package_name in target_package_names:
        molly_response = execute_one(get_molly().start_scan(get_molly_params(package_name, target_env))).json()
        scan_ids += molly_response['scan_id']

    release_ticket = find_release_ticket(conductor_request['comment'])
    if release_ticket:
        attach_molly_report_links(release_ticket, scan_ids)
        # TODO: implement scan status reporting
        # set_molly_scan_status(release_ticket, 'started')
        # init_scan_status_polling(release_ticket, scan_ids) -> set_molly_scan_status(release_ticket, 'finished')
    return scan_ids


def conductor_package_deploying_finished(conductor_request, service_id, environment_id):
    if not is_final_call(conductor_request):
        return
    target_package_names = [pkg_description['package'] for pkg_description in conductor_request['packages']]
    target_env = conductor_request['branch']
    if target_env not in ['stable', 'prestable']:
        logging.debug('Package deployed to branch "{}". Do nothing'.format(target_env))
        return

    ongoing_events = get_infra().get_events(
        service_id=service_id,
        environment_id=environment_id,
    )
    finish_time = int(time())
    for package_name in target_package_names:
        for event in ongoing_events:
            get_infra().update_event(event['id'], finishTime=finish_time)
