import logging

from passport.backend.pocket_service.utils.github import get_github
from passport.backend.pocket_service.webhooks.common import (
    do_transition,
    find_tickets,
    find_tickets_for_transition,
    url_to_ticket,
)


def teamcity_url_for_pull_request(project, pr_no):
    project_to_type = {
        'passport-api': 'PassportPython_PassportApiPrecise64templated',
        'passport-core': 'PassportPython_PassportCorePrecise64templated',
        'passport-settings': 'PassportPython_PassportSettingsPrecise64pr',
        'vault': 'PassportPython_PassportVault',
    }

    try:
        type_id = project_to_type[project]
    except KeyError:
        logging.debug('Cannot make teamcity url for project {project}.'.format(project=project))
        return

    return 'https://teamcity.yandex-team.ru/viewType.html?buildTypeId={type_id}&branch_PassportPython={pr_no}%2Fmerge&tab=buildTypeStatusDiv'.format(
        type_id=type_id,
        pr_no=pr_no,
    )


def get_rebase_url(owner, project, pr_no):
    return 'https://passport-pocket-service.n.yandex-team.ru/passport/rebase/{}/{}/{}'.format(owner, project, pr_no)


def github_resolve_on_pr_merge(github_request):
    if 'pull_request' not in github_request:
        return
    pull_request_url = github_request['pull_request']['url']
    if github_request['action'] != 'closed':
        logging.debug('PR {} is not closed, skipping'.format(pull_request_url))
        return
    if not github_request['pull_request']['merged']:
        logging.debug('PR {} is not merged, skipping'.format(pull_request_url))
        return

    pull_request_title = github_request['pull_request']['title']

    tickets = find_tickets(pull_request_title)
    tickets_for_transition = find_tickets_for_transition(tickets, 'resolved')
    return do_transition(tickets_for_transition, 'resolved')


def github_add_links_on_pr_open(github_request, include_tickets=True, include_teamcity=True, include_rebase=True):
    if 'pull_request' not in github_request:
        return
    pull_request_url = github_request['pull_request']['url']
    if github_request['action'] != 'opened':
        logging.debug('PR {} is not opened, skipping'.format(pull_request_url))
        return

    pull_request_number = github_request['pull_request']['number']

    pull_request_title = github_request['pull_request']['title']
    pull_request_body = github_request['pull_request']['body']

    # находим тикеты для стартрека
    tickets = set(find_tickets(pull_request_title) + find_tickets(pull_request_body))
    tickets = {t: url_to_ticket(t) for t in tickets}

    # конструируем ссылку на тимсити
    current_project = github_request['repository']['name']
    full_repo_name = github_request['repository']['full_name']
    teamcity_url = teamcity_url_for_pull_request(current_project, pull_request_number)

    rebase_url = get_rebase_url(full_repo_name.split('/')[0], current_project, pull_request_number)

    new_pull_request_body = []

    if include_tickets and tickets:
        tickets_text = []
        for ticket, ticket_url in sorted(tickets.items(), key=lambda kv: kv[0]):
            tickets_text.append('[{}]({})'.format(ticket, ticket_url))
        tickets_text = ', '.join(tickets_text)
        new_pull_request_body.append(tickets_text)
    if include_teamcity and teamcity_url:
        new_pull_request_body.append(u'[Сборка на тимсити]({})'.format(teamcity_url))
    if include_rebase:
        new_pull_request_body.append(u'[Отребейзить]({})'.format(rebase_url))

    if any([include_tickets, include_teamcity, include_rebase]) and new_pull_request_body:
        new_pull_request_body = pull_request_body + '\n\n' + '&nbsp;&nbsp;&nbsp;&nbsp;'.join(new_pull_request_body)
    else:
        new_pull_request_body = pull_request_body

    github = get_github()
    github.update_pull_request(
        full_repo_name.split('/')[0],
        current_project,
        pull_request_number,
        data={
            'body': new_pull_request_body,
        },
    )

    return {
        'project': current_project,
        'repo': full_repo_name,
        'tickets': tickets,
        'teamcity_url': teamcity_url,
        'pr_no': pull_request_number,
        'new_pull_request_body': new_pull_request_body,
        'rebase_url': rebase_url,
        'include_tickets': include_tickets,
        'include_teamcity': include_teamcity,
        'include_rebase': include_rebase,
    }
