from passport.backend.pocket_service.utils.abc import get_abc
from passport.backend.pocket_service.utils.http import (
    get_verify,
    requests_retry_session,
)
from passport.backend.pocket_service.utils.startrek import get_startrek_client


DEFAULT_SHEDULE_SLUG = 'passport-backend'  # https://abc.yandex-team.ru/services/passp/duty/?role=1906
DEFAULT_AGILE_BOARD_ID = 2607  # https://st.yandex-team.ru/agile/board/2607


def _get_active_sprint(board_id):
    sprints = get_startrek_client().boards[board_id].sprints
    sprint = None
    for item in sprints:
        if item['status'] == 'in_progress':
            if sprint is not None:
                raise RuntimeError('Too many active sprints')
            sprint = item
    return sprint


def _who_is_on_duty_login(shedule_slug):
    request = get_abc().who_is_on_duty(shedule_slug=shedule_slug)
    response = requests_retry_session().send(request, verify=get_verify()).json()
    if not response:
        raise RuntimeError('No schedule found for given ABC service')
    if len(response) > 1:
        raise RuntimeError('Too many duties found')
    return response[0]['person']['login']


def _who_is_on_duty_v2_login(shedule_slug, shedule_id=None):
    if shedule_id is None:
        request = get_abc().get_duty_v2_schedule_id_by_slug(shedule_slug=shedule_slug)
        result = requests_retry_session().send(request, verify=get_verify()).json()['result']
        if len(result) == 0:
            raise RuntimeError('No schedule found for given slug')
        shedule_id = result[0]['id']

    request = get_abc().who_is_on_duty_v2(shedule_id=shedule_id)
    result = requests_retry_session().send(request, verify=get_verify()).json()['result']
    if len(result) == 0:
        raise RuntimeError('No duties found for given schedule')
    if len(result) > 1:
        raise RuntimeError('Too many duties found')
    return result[0]['staff']['login']


def assign_to_duty_person(startrek_request):
    issue_id = startrek_request['issue_id']
    shedule_id = startrek_request.get('shedule_id')
    shedule_slug = startrek_request.get('shedule_slug', DEFAULT_SHEDULE_SLUG)
    agile_board_id = startrek_request.get('agile_board_id', DEFAULT_AGILE_BOARD_ID)
    summon_message = startrek_request.get('summon_message')
    is_duty_v2 = startrek_request.get('is_duty_v2', False)

    if is_duty_v2:
        duty_login = _who_is_on_duty_v2_login(shedule_slug=shedule_slug, shedule_id=shedule_id)
    else:
        duty_login = _who_is_on_duty_login(shedule_slug=shedule_slug)

    active_sprint = _get_active_sprint(board_id=agile_board_id)
    if not active_sprint:
        raise RuntimeError('No active sprint found')

    issue = get_startrek_client().issues[issue_id]
    issue.update(
        assignee=duty_login,
        sprint=[active_sprint],
    )
    if summon_message:
        issue.comments.create(text=summon_message, summonees=[duty_login])


def set_current_sprint(startrek_request):
    issue_id = startrek_request['issue_id']
    agile_board_id = startrek_request.get('agile_board_id', DEFAULT_AGILE_BOARD_ID)

    active_sprint = _get_active_sprint(board_id=agile_board_id)
    if not active_sprint:
        raise RuntimeError('No active sprint found')

    issue = get_startrek_client().issues[issue_id]
    issue.update(
        sprint=[active_sprint],
    )
