# coding: utf-8
from datetime import (
    date,
    timedelta,
)

from passport.backend.library.commander import (
    ArgumentGroup,
    Commander,
)
from passport.backend.profile.configs import config
from passport.backend.profile.jobs.blackbox_events import prepare_day_data_from_blackbox
from passport.backend.profile.jobs.calculate_auth_frequencies import create_userprofiles
from passport.backend.profile.jobs.oauth_events import prepare_day_data_from_oauth
from passport.backend.profile.jobs.passport_events import prepare_day_data_from_passport
from passport.backend.profile.jobs.passport_glogouts_events import prepare_glogouts_day_data_from_passport
from passport.backend.profile.scripts.build_profile_daily import build_profile_daily
from passport.backend.profile.scripts.cleanup import cleanup
from passport.backend.profile.scripts.create_profile_tables import (
    create_profile_tables,
    drop_profile_tables,
)
from passport.backend.profile.scripts.delete_old_rows import delete_old_rows
from passport.backend.profile.scripts.upload_profile_to_ydb import upload_profile_to_ydb
from passport.backend.profile.utils.helpers import from_str_to_date


def main():
    config.set_logging()
    commander = Commander()

    argument_group = ArgumentGroup().add_argument(
        '--target-date',
        dest='target_date',
        nargs='?',
        type=lambda string: from_str_to_date(string).date(),
        default=date.today() - timedelta(days=1),
    )

    argument_group_with_rerun = ArgumentGroup().add_argument(
        '--target-date',
        dest='target_date',
        nargs='?',
        type=lambda string: from_str_to_date(string).date(),
        default=date.today() - timedelta(days=1),
    ).add_argument(
        '--force-rerun',
        action='store_true',
        default=False,
        dest='force_rerun',
    )

    commander.add_command(
        'build-profile-daily',
        build_profile_daily,
        config=config,
    ).add_group(argument_group_with_rerun)

    commander.add_command(
        'cleanup',
        cleanup,
        config=config,
    )

    commander.add_command(
        'prepare-day-data-from-blackbox',
        prepare_day_data_from_blackbox,
        config=config,
    ).add_group(argument_group)

    commander.add_command(
        'prepare-day-data-from-passport',
        prepare_day_data_from_passport,
        config=config,
    ).add_group(argument_group)

    commander.add_command(
        'prepare-glogouts-day-data-from-passport',
        prepare_glogouts_day_data_from_passport,
        config=config,
    ).add_group(argument_group)

    commander.add_command(
        'prepare-day-data-from-oauth',
        prepare_day_data_from_oauth,
        config=config,
    ).add_group(argument_group)

    commander.add_command(
        'create-userprofiles',
        create_userprofiles,
        config=config,
    ).add_group(argument_group)

    commander.add_command(
        'create-profile-tables',
        create_profile_tables,
        config=config,
    )

    commander.add_command(
        'delete-old-rows',
        delete_old_rows,
        config=config,
    )

    commander.add_command(
        'drop-profile-tables',
        drop_profile_tables,
        config=config,
    )

    commander.add_command(
        'upload-profile-to-ydb',
        upload_profile_to_ydb,
        config=config,
    ).add_group(argument_group_with_rerun)

    commander.invoke()
