# -*- coding: utf-8 -*-
from datetime import datetime

from passport.backend.profile.utils.helpers import (
    get_host,
    Weekday,
)
from passport.backend.profile.utils.parsers import (
    parse_uatraits_bool_missing,
    parse_uatraits_value,
)
from qb2.api.v1 import (
    extractors as se,
    resources as sr,
)
from qb2_extensions.utils.geo import RegionTypes


def useragent(field='user_agent'):
    """
    Преобразования user_agent
    """
    return [
        se.custom('parsed_os', lambda detector, user_agent: detector.detect(user_agent), sr.resource("UATraits"), field).allow_override().hide(),
        se.custom('browser_name', lambda parsed_os: parse_uatraits_value(parsed_os.get('BrowserName')), 'parsed_os').allow_override(),
        se.custom('browser_version', lambda parsed_os: parse_uatraits_value(parsed_os.get('BrowserVersion')), 'parsed_os').allow_override(),
        se.custom('os_family', lambda parsed_os: parse_uatraits_value(parsed_os.get('OSFamily')), 'parsed_os').allow_override(),
        se.custom('os_name', lambda parsed_os: parse_uatraits_value(parsed_os.get('OSName', parsed_os.get('OSFamily'))), 'parsed_os').allow_override(),
        se.custom('os_version', lambda parsed_os: parse_uatraits_value(parsed_os.get('OSVersion')), 'parsed_os').allow_override(),
        se.custom('is_mobile', lambda parsed_os: int(parse_uatraits_bool_missing(parsed_os.get('isMobile'))), 'parsed_os').allow_override(),
    ]


def date_and_time(name='datetime_features', field='unixtime'):
    """
    Преобразования даты и времени
    """
    return [
        se.custom(name, _prepare_features_from_unixtime, field).hide(),
        se.dictitem('day_part', name).allow_override(),
        se.dictitem('weekday', name).allow_override(),
        se.dictitem('is_weekend', name).allow_override(),
        se.dictitem('hour', name).allow_override(),
        se.dictitem('month', name).allow_override(),
    ]


def geo(field='ip'):
    """
    Преобразования ip адреса в местоположение по геобазе
    """
    return [
        se.custom('region', lambda geobase, ip: geobase.region_by_ip(ip), sr.resource("Geobase"), field).allow_override().hide(),
        se.custom('is_black_ip', lambda region: int(region.is_black), 'region'),
        se.attribute('geo_id', 'id', from_='region').allow_override(),
        se.custom('country_region', lambda region: _get_parent_region_by_type(region, RegionTypes.COUNTRY), 'region').hide(),
        se.custom('city_region', lambda region: _get_parent_region_by_type(region, RegionTypes.CITY), 'region').hide(),
        se.attribute('country_id', 'id', from_='country_region'),
        se.attribute('city_id', 'id', from_='city_region'),
        se.custom('as_list', lambda ip_origin, ip: ip_origin.region_by_ip(ip), sr.resource('IpOrigins'), field),
    ]


def canonized_host(name, field, parts=None):
    return se.custom(name, lambda url: get_host(url, parts) if url else None, field)


def _prepare_features_from_unixtime(unixtime):
    date_time = datetime.fromtimestamp(int(unixtime))
    weekday = date_time.isoweekday()
    hour = date_time.hour
    month = date_time.month

    return {
        'day_part': int(hour / 6),  # делим сутки на 4 части (утро, день, вечер, ночь)
        'weekday': weekday,
        'is_weekend': int(weekday in (Weekday.SATURDAY, Weekday.SUNDAY)),  # нужен ли?
        'hour': hour,
        'month': month,
    }


def _get_parent_region_by_type(region, region_type):
    return _get_parent_region_by_condition(region, lambda r: r.type == region_type)


def _get_parent_region_by_condition(region, condition):
    if condition(region):
        return region

    for parent_region in reversed(region.path):
        if condition(parent_region):
            return parent_region

    return None
