# -*- coding: utf-8 -*-
import os

from passport.backend.profile import (
    extractors as pe,
    get_cluster,
)
from passport.backend.profile.utils.helpers import (
    cleanup_ip,
    to_date_str,
)
from passport.backend.profile.utils.yt import (
    check_table_attribute_exist,
    set_table_attribute,
)
from qb2.api.v1 import (
    extractors as se,
    filters as sf,
    typing,
)
from retrying import retry
from yt.wrapper.errors import YtIncorrectResponse


BLACKBOX_DATASET_JOB_STATUS_ATTRIBUTE = 'blackbox_dataset_job_finished'


@retry(stop_max_attempt_number=3, wait_fixed=5000, retry_on_exception=(YtIncorrectResponse,))
def prepare_day_data_from_blackbox(config, target_date):
    blackbox_dataset_path = os.path.join(config['yt']['blackbox_dataset_dir'], to_date_str(target_date))
    if check_table_attribute_exist(config, blackbox_dataset_path, BLACKBOX_DATASET_JOB_STATUS_ATTRIBUTE):
        return

    cluster = get_cluster(config).env()

    job = cluster.job(name='prepare-day-data-from-blackbox-' + to_date_str(target_date))
    blackbox_log = job.table(os.path.join(config['yt']['blackbox_log_dir'], to_date_str(target_date)))

    fields = [
        se.log_field('action').allow_override().hide(),
        se.log_field('type').hide(),
        se.log_field('referer').hide(),
        se.log_field('retpath').hide(),
        se.log_field('status').hide(),
        se.log_field('useragent').hide(),
        se.log_field('proxy_ip').allow_null_dependency().hide(),
        se.log_field('user_ip').allow_null_dependency().hide(),
        se.integer_log_field('uid').allow_override(),
        se.integer_log_field('unixtime'),
        pe.canonized_host('retpath_host', 'retpath'),
        pe.canonized_host('referer_host', 'referer'),
        se.custom(
            'ip', lambda ip_proxy, user_ip: cleanup_ip(ip_proxy or user_ip), 'proxy_ip', 'user_ip'
        ).allow_override().allow_null_dependency(),
    ]
    fields.extend(pe.date_and_time())
    fields.extend(pe.useragent(field='useragent'))
    fields.extend(pe.geo())

    # выбираем события:
    #  - подновление сессии
    blackbox_events_log = blackbox_log.qb2(
        log='passport-log',
        fields=fields,
        filters=[
            sf.or_(
                # событие подновление сессии
                sf.and_(
                    sf.equals('action', 'auth'),
                    sf.equals('status', 'ses_update'),
                    sf.equals('type', 'web'),
                ),
            ),
        ],
        intensity='default',
    )
    blackbox_events_log.sort('uid').put(
        blackbox_dataset_path,
        schema=dict(
            as_list=typing.Json,
            browser_name=str,
            browser_version=str,
            city_id=int,
            country_id=int,
            day_part=int,
            geo_id=int,
            hour=int,
            ip=str,
            is_black_ip=int,
            is_mobile=int,
            is_weekend=int,
            month=int,
            os_family=str,
            os_name=str,
            os_version=str,
            referer_host=str,
            retpath_host=str,
            uid=int,
            unixtime=int,
            weekday=int,
        )
    )
    job.run()

    set_table_attribute(config, blackbox_dataset_path, BLACKBOX_DATASET_JOB_STATUS_ATTRIBUTE, True)
