# -*- coding: utf-8 -*-
import os
import re

from passport.backend.profile import (
    extractors as pe,
    get_cluster,
)
from passport.backend.profile.utils.helpers import (
    to_date_str,
    truncate_am_version,
)
from passport.backend.profile.utils.yt import (
    check_table_attribute_exist,
    set_table_attribute,
)
from qb2.api.v1 import (
    extractors as se,
    filters as sf,
    typing,
)
from retrying import retry
from yt.wrapper.errors import YtIncorrectResponse


OAUTH_DATASET_JOB_STATUS_ATTRIBUTE = 'oauth_dataset_job_finished'


@retry(stop_max_attempt_number=3, wait_fixed=5000, retry_on_exception=(YtIncorrectResponse,))
def prepare_day_data_from_oauth(config, target_date):
    oauth_dataset_path = os.path.join(config['yt']['oauth_dataset_dir'], to_date_str(target_date))
    if check_table_attribute_exist(config, oauth_dataset_path, OAUTH_DATASET_JOB_STATUS_ATTRIBUTE):
        return
    cluster = get_cluster(config).env()

    job = cluster.job(name='prepare-day-data-from-oauth-' + to_date_str(target_date))
    oauth_log = job.table(os.path.join(config['yt']['oauth_log_dir'], to_date_str(target_date)))

    fields = [
        se.log_field('action').allow_override().hide(),
        se.log_field('mode').hide(),
        se.log_field('status').hide(),
        se.log_field('grant_type').hide(),
        se.log_field('user_ip'),
        se.log_field('am_version').hide(),
        se.log_field('device_id'),
        se.log_field('cloud_token'),
        se.integer_log_field('unixtime'),

        se.integer_log_field('uid').allow_override(),
        se.custom(
            'am_version_truncated', lambda am_version: truncate_am_version(am_version), 'am_version',
        ).allow_override().allow_null_dependency(),
    ]
    fields.extend(pe.date_and_time())
    fields.extend(pe.geo(field='user_ip'))

    oauth_log = oauth_log.qb2(
        log='generic-tskv-log',
        fields=fields,
        filters=[
            sf.or_(
                # событие получения токена
                sf.and_(
                    sf.equals('mode', 'issue_token'),
                    sf.equals('action', 'issue'),
                    sf.equals('status', 'ok'),
                    sf.not_(sf.equals('grant_type', 'password')),  # будет ликвидировано, проверять нет нужды

                    sf.defined('am_version', 'am_version_truncated', 'device_id'),  # мусорные или пустые device_id и am_version не должны попадать в профиль
                    sf.custom(lambda device_id: re.match('^([ -~]){6,50}$', device_id) is not None),
                ),
            ),
        ],
        intensity='default',
    )
    oauth_log.sort('uid').put(
        oauth_dataset_path,
        schema=dict(
            am_version_truncated=str,
            as_list=typing.Json,
            city_id=int,
            cloud_token=str,
            country_id=int,
            day_part=int,
            device_id=str,
            geo_id=int,
            hour=int,
            is_black_ip=int,
            is_weekend=int,
            month=int,
            uid=int,
            unixtime=int,
            user_ip=str,
            weekday=int,
        )
    )
    job.run()

    set_table_attribute(config, oauth_dataset_path, OAUTH_DATASET_JOB_STATUS_ATTRIBUTE, True)
