# -*- coding: utf-8 -*-
import os

from nile.api.v1 import Record
from passport.backend.profile import (
    extractors as pe,
    get_cluster,
)
from passport.backend.profile.utils.helpers import to_date_str
from passport.backend.profile.utils.yt import (
    check_table_attribute_exist,
    set_table_attribute,
)
from qb2.api.v1 import (
    extractors as se,
    filters as sf,
    typing,
)
from retrying import retry
from yt.wrapper.errors import YtIncorrectResponse


PASSPORT_DATASET_JOB_STATUS_ATTRIBUTE = 'passport_dataset_job_finished'


def track_reducer(groups):
    for track_id, records in groups:
        submitted_record = successful_auth_record = None

        for record in records:
            if submitted_record is None and (
                record['action'] == 'submitted' or
                record['action'] == 'decision_reached'
            ):
                submitted_record = record
            if successful_auth_record is None and record['action'] == 'cookie_set':
                successful_auth_record = record

        if not submitted_record:
            # если нет записи action=submitted, то это не авторизация
            continue

        if successful_auth_record:
            record_dict = successful_auth_record.to_dict()
            del record_dict['action']
            del record_dict['track_id']
            record_dict['retpath_host'] = record_dict.get('retpath_host') or submitted_record.get('retpath_host')
            record_dict['referer_host'] = record_dict.get('referer_host') or submitted_record.get('referer_host')
            yield Record(**record_dict)


@retry(stop_max_attempt_number=3, wait_fixed=5000, retry_on_exception=(YtIncorrectResponse,))
def prepare_day_data_from_passport(config, target_date):
    passport_dataset_path = os.path.join(config['yt']['passport_dataset_dir'], to_date_str(target_date))
    if check_table_attribute_exist(config, passport_dataset_path, PASSPORT_DATASET_JOB_STATUS_ATTRIBUTE):
        return
    cluster = get_cluster(config).env()

    job = cluster.job(name='prepare-day-data-from-passport-' + to_date_str(target_date))
    passport_log = job.table(os.path.join(config['yt']['passport_log_dir'], to_date_str(target_date)))

    fields = [
        'track_id',
        se.log_field('action').allow_override(),
        se.log_field('type').hide(),
        se.log_field('retpath').hide(),
        se.log_field('referer').hide(),
        se.log_field('user_agent').allow_override().hide(),
        se.log_field('is_auth_challenge_shown').hide(),
        se.log_field('ip').allow_override(),
        se.log_field('yandexuid').allow_override(),
        se.log_field('track_type').hide(),
        se.integer_log_field('unixtime'),
        se.integer_log_field('captcha_passed'),

        se.integer_log_field('uid').allow_override(),
        se.custom('is_challenge_shown', lambda x: int(x) if x is not None else 0, 'is_auth_challenge_shown'),
        pe.canonized_host('retpath_host', 'retpath'),
        pe.canonized_host('referer_host', 'referer'),
    ]
    fields.extend(pe.date_and_time())
    fields.extend(pe.useragent())
    fields.extend(pe.geo())

    # выбираем события проверки паролей
    auth_log = passport_log.qb2(
        log='passport-log',
        fields=fields,
        filters=[
            sf.or_(
                # событие отправки авторизационной формы
                sf.and_(
                    sf.equals('mode', 'any_auth'),
                    sf.equals('action', 'submitted'),
                    sf.or_(
                        sf.equals('type', 'password'),
                        sf.equals('type', 'multi_step'),
                    ),
                ),
                # успешные авторизации после прохождения challenge
                sf.and_(
                    sf.equals('mode', 'any_auth'),
                    sf.equals('action', 'cookie_set'),
                ),
                sf.and_(
                    sf.equals('mode', 'any_auth'),
                    sf.equals('action', 'decision_reached'),
                    sf.equals('track_type', 'authorize'),
                ),
            ),
        ],
        intensity='default',
    ).groupby('track_id').reduce(track_reducer)
    auth_log.sort('uid').put(
        passport_dataset_path,
        schema=dict(
            as_list=typing.Json,
            browser_name=str,
            browser_version=str,
            captcha_passed=int,
            city_id=int,
            country_id=int,
            day_part=int,
            geo_id=int,
            hour=int,
            ip=str,
            is_black_ip=int,
            is_challenge_shown=int,
            is_mobile=int,
            is_weekend=int,
            month=int,
            os_family=str,
            os_name=str,
            os_version=str,
            referer_host=str,
            retpath_host=str,
            uid=int,
            unixtime=int,
            weekday=int,
            yandexuid=str
        )
    )
    job.run()

    set_table_attribute(config, passport_dataset_path, PASSPORT_DATASET_JOB_STATUS_ATTRIBUTE, True)
