# -*- coding: utf-8 -*-
import datetime
import os

from nile.api.v1 import Record
from passport.backend.profile import get_cluster
from passport.backend.profile.utils.helpers import (
    date_to_integer_unixtime,
    to_date_str,
)
from passport.backend.profile.utils.yt import (
    check_table_attribute_exist,
    get_yt,
    set_table_attribute,
)
from qb2.api.v1 import (
    extractors as se,
    filters as sf,
)
from retrying import retry
from yt.wrapper.errors import YtIncorrectResponse


PASSPORT_DATASET_JOB_STATUS_ATTRIBUTE = 'passport_glogouts_dataset_job_finished'


def last_logout_reducer(half_a_year_ago_unixtime):
    def _last_logout_reducer(groups):
        for uid, records in groups:
            max_time = max([record['glogout_unixtime'] for record in records])
            # если последний глогаут был более полгода назад перестаем это помнить, так как профиль содержит всего полгода историю
            if max_time < half_a_year_ago_unixtime:
                continue
            yield Record(
                uid=uid.uid,
                glogout_unixtime=max_time
            )

    return _last_logout_reducer


@retry(stop_max_attempt_number=3, wait_fixed=5000, retry_on_exception=(YtIncorrectResponse,))
def prepare_glogouts_day_data_from_passport(config, target_date):
    passport_glogouts_dataset_path = os.path.join(config['yt']['passport_glogouts_dataset_dir'], to_date_str(target_date))
    passport_glogouts_dataset_yesterday_path = os.path.join(config['yt']['passport_glogouts_dataset_dir'], to_date_str(target_date-datetime.timedelta(days=1)))
    if check_table_attribute_exist(config, passport_glogouts_dataset_path, PASSPORT_DATASET_JOB_STATUS_ATTRIBUTE):
        return
    cluster = get_cluster(config).env()
    yt = get_yt(config)

    job = cluster.job(name='prepare-day-glogouts-from-passport-' + to_date_str(target_date))
    passport_log = job.table(os.path.join(config['yt']['passport_log_dir'], to_date_str(target_date)))

    # выбираем события последнего глогаута пользователей за текущий день
    glogout_log = passport_log.qb2(
        log='passport-log',
        fields=[
            se.integer_log_field('uid'),
            se.integer_log_field('unixtime').hide(),
            se.copy('glogout_unixtime', 'unixtime')
        ],
        filters=[
            sf.equals('entity', 'account.global_logout_datetime'),
        ],
        intensity='default',
    )

    # таблица имеет накопительный эффект, если есть вчерашняя то прибавим ее к сегодняшней
    if yt.exists(passport_glogouts_dataset_yesterday_path):
        glogout_log = glogout_log.concat(job.table(passport_glogouts_dataset_yesterday_path))

    half_a_year_ago_unixtime = date_to_integer_unixtime(target_date - datetime.timedelta(days=366 / 2))
    glogout_log = glogout_log.groupby('uid').reduce(last_logout_reducer(half_a_year_ago_unixtime))
    glogout_log.sort('uid').put(
        passport_glogouts_dataset_path,
        schema=dict(
            uid=int,
            glogout_unixtime=int,
        )
    )
    job.run()

    set_table_attribute(config, passport_glogouts_dataset_path, PASSPORT_DATASET_JOB_STATUS_ATTRIBUTE, True)
