# coding: utf-8
from contextlib import contextmanager
from datetime import (
    date,
    timedelta,
)
import json
import logging
import os
import signal
import sys
import time

from passport.backend.clients.mr import YtClient
from passport.backend.profile.configs import get_config
from passport.backend.profile.monitor.redirect_stdout import redirect_stdout_stderr_to_logger
from passport.backend.utils.time import to_date_str


PASSPORT_ROBOT_LOGIN = 'robot-pssp-profile'
FLAP_SUPPRESSION_TIME = 20  # минуты
MAX_CHECK_TIMEOUT = 27  # секунд на внешние запросы


@contextmanager
def timeout(sec):
    """
        Если действия внутри будут идти более установленного кол-ва секунд будет выкинуто исключение TimeoutError
    """
    signal.signal(signal.SIGALRM, raise_timeout)
    signal.alarm(sec)
    yield
    signal.signal(signal.SIGALRM, signal.SIG_IGN)


class TimeoutError(Exception):
    pass


def raise_timeout(signum, frame):
    raise TimeoutError()


def _save_and_return_status(tmp_dir, prev_status_filename, current_status):
    # сохраняем предыдущий статус и время его смены в файл, чтоб подавить флапы, типа неответа YT
    # формат {'status': '...',  # последний полученный статус
    #         'prev_status': '...',  # предыдущий стабильный статус
    #         'time': 12345678}  # время смены между ними
    current_time = int(time.time())
    data_to_save = {'status': current_status,
                    'prev_status': '0;ok',
                    'time': current_time}
    if not os.path.isdir(tmp_dir):
        os.makedirs(tmp_dir)
    filepath = os.path.join(tmp_dir, prev_status_filename)
    is_new_file = not os.path.exists(filepath)
    mode = 'w' if is_new_file else 'r+'
    with open(filepath, mode) as f:
        if not is_new_file:
            prev_state = json.loads(f.read())
            data_to_save = prev_state.copy()

            prev_status_str = prev_state.get('prev_status', '0;ok')
            prev_status_int = int(prev_status_str[0])
            last_status_str = prev_state.get('status', '0;ok')
            last_status_int = int(last_status_str[0])
            change_time = prev_state.get('time', current_time)

            if current_time - change_time > FLAP_SUPPRESSION_TIME * 60:  # 10 минут прошло со смены статуа
                if int(current_status[0]) == last_status_int:  # и последний наш статус так и нее поменялся
                    print(current_status)  # отдаем его, ничего снова сохранять в файл не надо
                    return
                else:
                    data_to_save['prev_status'] = last_status_str
                    data_to_save['status'] = current_status
                    data_to_save['time'] = current_time
                    print(last_status_str)
            else:  # прошло менее 10 минут со смены статуса
                if int(current_status[0]) == prev_status_int:  # статус вернулся к предыдущему значению, скидываем таймер
                    data_to_save['prev_status'] = current_status
                    data_to_save['status'] = current_status
                    data_to_save['time'] = current_time
                    print(prev_status_str)
                else:  # статус еще не устоялся, шлем предыдущий и ждем до истечения 10 мин
                    print(prev_status_str)
                    return
        else:
            print(current_status)

    with open(filepath, 'w') as f:
        f.write(json.dumps(data_to_save))


def main():
    # Мониторинг ходит/проверяет исключительно прод
    config = get_config(environment='production')
    config.set_logging()
    log = logging.getLogger('passport.profile.monitor')

    # все выводы YT и модулей под ним перенаправляются в наши же логи, иначе может написать в stderr при ретрае и проверка упадет, даже если в stdout будет верный ответ
    root_logger = logging.getLogger()
    for library in ('requests', 'urllib3', 'yt', 'Yt'):
        logger = logging.getLogger(library)
        logger.handlers = list(root_logger.handlers)
        logger.setLevel(logging.WARNING)

    def save_and_return_status(status=None, default_status='0;ok'):
        _save_and_return_status(config['yt']['tmp_dir'], config['yt']['prev_status_filename'], status or default_status)

    day_before = to_date_str(date.today() - timedelta(days=1))
    day_before_yesterday = to_date_str(date.today() - timedelta(days=2))
    # таблица профиля текущего дня в yt
    yt_table_path = os.path.join(config['yt']['profile_dir'], day_before)
    yt_table_path_yesterday = os.path.join(config['yt']['profile_dir'], day_before_yesterday)
    # пути к таблицам yt построения профиля
    yt_table_path_list = [os.path.join(config['yt'][table], day_before) for table in ['blackbox_dataset_dir', 'passport_dataset_dir', 'oauth_dataset_dir', 'profile_dir']]
    try:
        with redirect_stdout_stderr_to_logger(log):
            with timeout(MAX_CHECK_TIMEOUT):
                yt = YtClient(
                    proxy=config['yt']['cluster_name'],
                    token=config['yt']['token'],
                    spec={'proxy': {'connect_timeout': 2000,
                                    'request_timeout': 5000,
                                    'retries': {
                                        'enable': True,
                                        'count': 3,  # по умолчанию 6
                                        'total_timeout': 15000,
                                        'backoff': {
                                            'policy': 'constant_time',
                                            'constant_time': 2000,
                                        }}
                                    }}
                )

                # запущенные операции в YT
                operations_today = yt.list_operations(user=PASSPORT_ROBOT_LOGIN,
                                                      from_time=day_before + 'T00:00:00.000000Z',
                                                      state='running',
                                                      filter='Uploading profile ',
                                                      include_counters=False,
                                                      )['operations']

                # наличие таблиц построения профиля в yt
                yt_table_exist = [yt.exists(path) for path in yt_table_path_list]

                # если аттрибут есть на таблице - создание профилей в YT завершено успешно
                daily_job_status = yt.get_attribute(
                    yt_table_path,
                    'profile_daily_job_finished',
                    False
                )

                # если аттрибут есть на таблице - переливка профилей в YDB завершена успешно
                ydb_upload = yt.get_attribute(
                    yt_table_path,
                    'ydb_upload_job_status',
                    False
                )

                # если аттрибут есть на таблице - переливка профилей в YDB вчера завершена успешно
                ydb_upload_yesterday = yt.get_attribute(
                    yt_table_path_yesterday,
                    'ydb_upload_job_status',
                    False
                )
    except Exception as exc:
        log.error('Exception while checking profile job status', exc_info=exc)
        save_and_return_status('1;yt_execution_exception')
        sys.exit()

    try:
        # Безусловная проверка (не зависит от времени суток):
        # профиль загружен в ydb
        if ydb_upload:
            save_and_return_status()
            sys.exit()

        # идет ли загрузка профиля в YDB
        ydb_upload_title = 'Uploading profile {{date}} from {profile_dir}/{{date}} to {ydb_db}/{ydb_table}'.format(profile_dir=config['yt']['profile_dir'],
                                                                                                                   ydb_db=config['ydb']['database'],
                                                                                                                   ydb_table=config['ydb']['table_name'])
        ydb_upload_operation = ([operation for operation in operations_today if operation['brief_spec']['title'].startswith(ydb_upload_title.format(date=day_before))] or [None])[0]
        ydb_upload_in_process_from_yesterday = any([operation['brief_spec']['title'].startswith(ydb_upload_title.format(date=day_before_yesterday)) for operation in operations_today])

        # Проверки которые зависят от времени дня:
        if time.localtime().tm_hour < 6:
            # Если вчерашняя заливка профилей не завершилась и не идет сейчас
            if not ydb_upload_yesterday and not ydb_upload_in_process_from_yesterday:
                save_and_return_status('2;ydb_upload_failed_yesterday')
            else:
                save_and_return_status()
            sys.exit()

        elif 6 <= time.localtime().tm_hour < 12:
            # Если создание профиля в YT начато, то ждем до 12 часов
            if not any(yt_table_exist):
                save_and_return_status('2;profile_daily_job_failed')
            # вчерашняя таска загрузки в ydb не была завершена
            elif ydb_upload_in_process_from_yesterday:
                save_and_return_status('1;previous_upload_not_finished')
            else:
                save_and_return_status()
            sys.exit()

        elif 12 <= time.localtime().tm_hour < 18:
            # создание профиля закончено и идет загрузка в ydb
            if daily_job_status and ydb_upload_operation is not None:
                save_and_return_status()
            # профиль подготовлен в YT, но не началась заливка в YDB (нет джобы в состоянии running)
            elif daily_job_status and ydb_upload_operation is None:
                save_and_return_status('2;ydb_upload_failed')
            # Если создание профиля в YT не завершено, но на финальной стадии (логи обработаны, таблица профиля создана), то ждем до 18 часов
            elif all(yt_table_exist):
                save_and_return_status()
            else:
                save_and_return_status('2;profile_daily_job_failed')
            sys.exit()

        elif 18 <= time.localtime().tm_hour:
            # профили в YT уже должны быть все построены
            if not daily_job_status:
                save_and_return_status('2;profile_daily_job_failed')
            # заливка в YDB или идет или уже закончилась
            elif ydb_upload_operation is not None or ydb_upload:
                save_and_return_status()
            else:
                save_and_return_status('2;ydb_upload_failed')
            sys.exit()
    except Exception as exc:
        log.error('Exception while processing profile job status', exc_info=exc)
        save_and_return_status('1;processing_exception')
        sys.exit()

    # если дошли до этого места, то что-то не так с мониторингом, не замалчиваем эту ситуацию
    print('1;monitoring_unexpected_situation')
    sys.exit()
