# -*- coding: utf-8 -*-

import logging
import os
import time

from passport.backend.profile.jobs.blackbox_events import prepare_day_data_from_blackbox
from passport.backend.profile.jobs.calculate_auth_frequencies import create_userprofiles
from passport.backend.profile.jobs.oauth_events import prepare_day_data_from_oauth
from passport.backend.profile.jobs.passport_events import prepare_day_data_from_passport
from passport.backend.profile.utils.helpers import to_date_str
from passport.backend.profile.utils.yt import (
    ExclusiveLock,
    get_yt,
)
from retrying import retry
from yt.wrapper.errors import YtIncorrectResponse


log = logging.getLogger('passport.profile.scripts.build_profile_daily')


DAILY_JOB_STATUS_ATTRIBUTE = 'profile_daily_job_status'


@retry(stop_max_attempt_number=3, wait_fixed=5000, retry_on_exception=(YtIncorrectResponse,))
def get_profile_job_status(config, target_date):
    yt = get_yt(config=config)

    target_date = to_date_str(target_date)
    passport_log_path = os.path.join(config['yt']['passport_log_dir'], target_date)
    profile_path = os.path.join(config['yt']['profile_dir'], target_date)
    source_row_count = yt.row_count(passport_log_path)
    job_status = dict(
        job_started_source_row_count=source_row_count,
        job_started_timestamp=time.time(),
    )

    if not yt.exists(profile_path):
        log.info('Daily profile job (%s): profile not exists yet', target_date)
        return dict(job_status, need_rerun=True)

    prev_job_status = yt.get_attribute(profile_path, DAILY_JOB_STATUS_ATTRIBUTE, default={})
    if not prev_job_status.get('job_started_source_row_count'):
        log.info('Daily profile job (%s): profile job status not found in YT', target_date)
        return dict(job_status, need_rerun=True)

    prev_source_row_count = prev_job_status['job_started_source_row_count']
    if source_row_count > prev_source_row_count:
        log.info(
            'Daily profile job (%s): source log row count increased (cur %d, prev %d)',
            target_date,
            source_row_count,
            prev_source_row_count,
        )
    elif source_row_count == prev_source_row_count:
        log.info('Daily profile job (%s): source log row count unchanged (%s)', target_date, source_row_count)
    else:
        log.warning(
            'Daily profile job (%s): source log row count decreased (cur %d, prev %d)',
            target_date,
            source_row_count,
            prev_source_row_count,
        )

    return dict(job_status, need_rerun=source_row_count != prev_source_row_count)


@retry(stop_max_attempt_number=3, wait_fixed=5000, retry_on_exception=(YtIncorrectResponse,))
def save_profile_job_status(config, target_date, job_status):
    yt = get_yt(config=config)

    target_date = to_date_str(target_date)
    passport_log_path = os.path.join(config['yt']['passport_log_dir'], target_date)
    source_row_count = yt.row_count(passport_log_path)
    profile_path = os.path.join(config['yt']['profile_dir'], target_date)

    yt.set_attribute(
        profile_path,
        DAILY_JOB_STATUS_ATTRIBUTE,
        dict(
            job_status,
            job_finished_timestamp=time.time(),
            job_finished_source_row_count=source_row_count,
            # Если в дневных логах паспорта не менялось кол-во строк, то считаем что перезапуск не требуется (почему так? наверное осталось со времени когда обрабатывался один лог только)
            need_rerun=source_row_count != job_status['job_started_source_row_count'],
        ),
    )


def build_profile_daily(config, target_date, force_rerun):
    """
    Обновить профиль, используя данные прошедшего дня.
    @param force_rerun: требуется ли выполнить принудительный пересчет (auth-датасет, профиль).
    Если флаг не выставлен, ориентируемся на число строк в паспортном логе за новый день,
    при успешном завершении заливки сохраняем это число как атрибут профиля в YT.
    """
    with ExclusiveLock(
        config=config,
        lock_path=config['yt']['build_profile_daily_lock'] + '-' + to_date_str(target_date),
        log=log,
    ):
        job_status = get_profile_job_status(config=config, target_date=target_date)
        if not force_rerun and not job_status['need_rerun']:
            log.info('Daily profile job (%s): skipped', to_date_str(target_date))
            return
        log.info('Daily profile job (%s): started (force_rerun=%s)', to_date_str(target_date), force_rerun)

        prepare_day_data_from_passport(
            config=config,
            target_date=target_date,
        )
        log.info('Passport auth dataset calculated for %s', to_date_str(target_date))
        prepare_day_data_from_blackbox(
            config=config,
            target_date=target_date,
        )
        log.info('Blackbox auth dataset calculated for %s', to_date_str(target_date))
        prepare_day_data_from_oauth(
            config=config,
            target_date=target_date,
        )
        log.info('OAuth auth dataset calculated for %s', to_date_str(target_date))
        create_userprofiles(
            config=config,
            target_date=target_date,
        )
        log.info('Profiles calculated for %s', to_date_str(target_date))
        save_profile_job_status(config=config, target_date=target_date, job_status=job_status)
