# -*- coding: utf-8 -*-
import os

from passport.backend.profile import (
    extractors as pe,
    get_cluster,
)
from passport.backend.profile.utils.helpers import (
    cut_host,
    truncate_timestamp,
)
from qb2.api.v1 import (
    extractors as se,
    filters as sf,
)
from retrying import retry
from yt.wrapper.errors import YtIncorrectResponse


def auths_reducer(groups):
    for track_id, records in groups:
        failed_auth_record = None

        for record in records:
            if record['action'] == 'cookie_set':  # successful auth
                yield record
                break
            elif failed_auth_record is None and record['action'] in ('failed_auth', 'profile_threshold_exceeded'):   # failed auth
                failed_auth_record = record
        else:
            yield failed_auth_record


@retry(stop_max_attempt_number=3, wait_fixed=5000, retry_on_exception=(YtIncorrectResponse,))
def grep_user_auth(config, uid, date, input_dir, output_dir):
    cluster = get_cluster(config).env(date=date)

    job = cluster.job()
    passport_log = job.table(os.path.join(input_dir, '$date'))

    fields = [
        'track_id',
        se.log_fields(
            'action', 'type', 'yandexuid', 'user_agent', 'ip',
            'retpath', 'referer',
            'bruteforce', 'login_status', 'password_status',
        ),
        se.log_field('retpath').hide(),
        se.log_field('referer').hide(),
        se.integer_log_fields(
            'unixtime', 'uid', 'is_auth_challenge_shown',
            'is_password_change_required', 'captcha_passed',
        ),
        pe.canonized_host('retpath_host', 'retpath').hide(),
        pe.canonized_host('referer_host', 'referer').hide(),

        se.custom('retpath_host_2', lambda host: cut_host(host, 2), 'retpath_host'),
        se.custom('retpath_host_3', lambda host: cut_host(host, 3), 'retpath_host'),

        se.custom('referer_host_2', lambda host: cut_host(host, 2), 'referer_host'),
        se.custom('referer_host_3', lambda host: cut_host(host, 3), 'referer_host'),

        se.custom('unixtime_20min', lambda unixtime: truncate_timestamp(unixtime, 20 * 60), 'unixtime'),
    ]
    fields.extend(pe.date_and_time())
    fields.extend(pe.useragent())
    fields.extend(pe.geo())

    # выбираем события проверки паролей
    passport_log.qb2(
        log='passport-log',
        fields=fields,
        filters=[
            sf.equals('uid', uid),
            sf.equals('mode', 'any_auth'),
        ],
    ).groupby(
        'uid', 'ip', 'browser_name', 'browser_version', 'unixtime_20min',
    ).reduce(
        auths_reducer,
    ).put(output_dir)

    job.run()
