# -*- coding: utf-8 -*-
from datetime import timedelta
import logging
import os
import time

from passport.backend.profile import get_yql
from passport.backend.profile.utils.helpers import (
    date_to_integer_unixtime,
    to_date_str,
)
from passport.backend.profile.utils.yt import (
    ExclusiveLock,
    get_yt,
)


log = logging.getLogger('passport.profile.scripts.upload_profile_to_ydb')


def prepare_query(cluster_name, ydb_balancer, ydb_database, ydb_profile_table, ydb_upload_jobs, yt_profile_path, upload_unixtime_mcs):
    return '''
USE {cluster_name};

$ydb_endpoint = "{ydb_balancer}";
$ydb_database = "{ydb_database}";
$ydb_table = "{ydb_profile_table}";
$ydb_batch_size_bytes = 5242880;
$ydb_batch_size_rows = 10000;
$ydb_max_retries = 10000;

PRAGMA yt.DataSizePerJob = "600000000";
PRAGMA yt.UserSlots = "{ydb_upload_jobs}";

$script = @@
import cyson
import json

def convert_to_json(s, uid):
    processed = dict()
    for key, value in s.iteritems():
        processed[key] = cyson.loads(value)
    processed.update({{'uid' : uid,
                      'profile_type': 'full',
                      'timestamp' : {upload_unixtime_mcs} / 10 ** 6}})
    return json.dumps(processed)
@@;

$udf = Python2::convert_to_json(Callable<(Dict<String,String>, Uint64?)->Json>, $script);

SELECT
    SUM(Bytes) AS TotalBytes,
    SUM(Rows) AS TotalRows,
    SUM(Batches) AS TotalBatches,
    SUM(Retries) AS TotalRetries
FROM (
    PROCESS (
        SELECT
            Yson::ConvertToUint64(t.uid) as uid,
            cast(0 as uint64) as inverted_event_timestamp,
            cast(0 as uint64) as unique_id,
            $udf(_other, Yson::ConvertToUint64(t.uid)) as value,
            cast({upload_unixtime_mcs} as uint64) as updated_at
        FROM `{yt_profile_path}` as t
    )
    USING YDB::BulkPushData(
          TableRows()
        , $ydb_endpoint
        , $ydb_database
        , $ydb_table
        , AsTuple("token", SecureParam("token:default_kikimr"))
        , $ydb_batch_size_bytes
        , $ydb_batch_size_rows
        , $ydb_max_retries
    )
);'''.format(
        cluster_name=cluster_name,
        ydb_balancer=ydb_balancer,
        ydb_database=ydb_database,
        ydb_profile_table=ydb_profile_table,
        ydb_upload_jobs=ydb_upload_jobs,
        yt_profile_path=yt_profile_path,
        upload_unixtime_mcs=str(upload_unixtime_mcs),
    )


def upload_profile_to_ydb(config, target_date, force_rerun=False):
    with ExclusiveLock(config=config, lock_path=config['yt']['upload_profile_to_ydb_daily_lock'], log=log):
        yt_profile_path = os.path.join(config['yt']['profile_dir'], to_date_str(target_date))

        yt = get_yt(config=config)

        if not yt.exists(yt_profile_path):
            log.info('Upload job (%s): %s is not ready for upload', yt_profile_path, to_date_str(target_date))
            return

        upload_status = yt.get_attribute(yt_profile_path, 'ydb_upload_job_status', {})

        log.info(
            'Upload job to ydb status: %s',
            upload_status,
        )

        if upload_status.get('finished') and not force_rerun:
            return

        # Время полного профиля в ydb - последняя секунда того дня за который обсчитали профиль
        target_date_end_unixtime_mcs = (date_to_integer_unixtime(target_date + timedelta(days=1)) - 1) * 10 ** 6

        request = get_yql(config=config).query(
            title='Uploading profile %s from %s to %s (YQL)' % (
                target_date,
                yt_profile_path,
                config['ydb']['database'] + '/' + config['ydb']['table_name'],
            ),
            query=prepare_query(
                cluster_name=config['yt']['cluster_name'],
                ydb_database=config['ydb']['database'],
                ydb_balancer=config['ydb']['endpoint'],
                ydb_profile_table=config['ydb']['database'] + '/' + config['ydb']['table_name'],
                ydb_upload_jobs=config['ydb']['upload_jobs'],
                yt_profile_path=yt_profile_path,
                upload_unixtime_mcs=target_date_end_unixtime_mcs,
            ),
            syntax_version=1,
        )
        results = request.run().get_results(wait=True)
        log.info(
            'Upload job to ydb (%s): %s',
            to_date_str(target_date),
            results.status
        )
        if results.status != 'COMPLETED':
            raise Exception()

        upload_status['finished'] = True
        upload_status['job_finished_timestamp'] = time.time()
        yt.set_attribute(yt_profile_path, 'ydb_upload_job_status', upload_status)
