# -*- coding: utf-8 -*-
import datetime
from itertools import (
    chain,
    islice,
)
import time
import urlparse

from nile.api.v1 import Record
from passport.backend.core.utils.version import parse_am_version
from statbox_bindings2.string_utils.misc import canonize_vhost

from .parsers import parse_browser


DATE_FORMAT = '%Y-%m-%d'


class Weekday(object):
    MONDAY = 1
    TUESDAY = 2
    WEDNESDAY = 3
    THURSDAY = 4
    FRIDAY = 5
    SATURDAY = 6
    SUNDAY = 7


def find(func, iterable):
    result = filter(func, iterable)[:1]
    return result[0] if result else None


def browser_grouper(value, value_freq):
    current_browser_name, current_browser_version = parse_browser(value)
    if current_browser_name and current_browser_version:
        count = 0
        for browser, browser_count in value_freq:
            browser_name, browser_version = parse_browser(browser)
            if (browser_name == current_browser_name and
                    current_browser_version is not None and browser_version is not None and
                    current_browser_version >= browser_version):
                count += browser_count
        return count
    else:
        return 0


# IP, IP_DICT
def probability(value, value_freq, grouper=None):
    if not value_freq:
        return 0

    value_freq_dict = dict(value_freq)
    if isinstance(value, list) or isinstance(value, tuple):
        return float(sum([
            grouper(elem, value_freq) if grouper is not None else value_freq_dict.get(elem, 0)
            for elem in value
        ])) / sum(value_freq_dict.values())
    else:
        value_count = grouper(value, value_freq) if grouper is not None else value_freq_dict.get(value, 0)
        return float(value_count) / sum(value_freq_dict.values())


def probability_grouped(grouper):
    def wrapper(value, value_freq):
        return probability(value, value_freq, grouper)
    return wrapper


def to_date_str(date):
    return date.strftime(DATE_FORMAT)


def from_str_to_date(date):
    return datetime.datetime.strptime(date, DATE_FORMAT)


def date_to_integer_unixtime(date):
    return int(time.mktime(date.timetuple()))


def get_host(url, parts=None):
    try:
        host = canonize_vhost(urlparse.urlparse(url).netloc)
        if parts:
            # отрезаем домен нужного уровня
            host = '.'.join(host.split('.')[-1 * parts:])

        return host
    except ValueError:
        return


def cut_host(host, count):
    return '.'.join(host.split('.')[-1 * count:])


def merge_records(records):
    result_dict = {}
    for record in records:
        result_dict.update(record.to_dict())
    return Record(**result_dict)


def truncate_timestamp(unixtime, seconds):
    return unixtime - (unixtime % seconds)


def chunks(iterable, size):
    iterator = iter(iterable)
    for first in iterator:
        yield chain([first], islice(iterator, size - 1))


def cleanup_ip(ip):
    if ip and '.' in ip:
        return ip.replace('::ffff:', '')
    return ip


def truncate_am_version(am_version):
    """
    Версия AM содержит билд, уберем его, оставим только основные цифры версии
    """
    version = parse_am_version(am_version)
    if any(version):
        return '.'.join([str(x) for x in version])


class RewindableIterator(object):
    """
    Итератор, позволяющий перезапускать итерацию.
    """
    def __init__(self, iterable):
        self.iterator = iter(iterable)
        self.memorized_iter = None
        self.memorized_items = []

    def __iter__(self):
        return self

    def next(self):
        if self.memorized_iter is not None:
            try:
                return next(self.memorized_iter)
            except StopIteration:
                self.memorized_iter = None

        item = next(self.iterator)
        self.memorized_items.append(item)
        return item

    def rewind(self):
        self.memorized_iter = iter(self.memorized_items)
