# coding: utf-8
from __future__ import absolute_import

from contextlib import contextmanager

from retrying import retry
from yt.wrapper.client import YtClient
from yt.wrapper.errors import (
    YtConcurrentTransactionLockConflict,
    YtIncorrectResponse,
)


def get_yt(config):
    return YtClient(
        proxy=config['yt']['cluster_name'],
        token=config['yt']['token'],
    )


@contextmanager
def ExclusiveLock(config, lock_path, log=None):
    yt = get_yt(config)
    with yt.Transaction():
        try:
            if not yt.exists(lock_path):
                yt.create('document', lock_path, recursive=True)
            yt.lock(lock_path)
            yield
        except YtConcurrentTransactionLockConflict as e:
            if log is not None:
                log.info("%s can't be acquired" % lock_path)
            raise e
        else:
            if yt.exists(lock_path):
                yt.remove(lock_path)


@retry(stop_max_attempt_number=3, wait_fixed=5000, retry_on_exception=(YtIncorrectResponse,))
def check_table_attribute_exist(config, table_path, attribute_name):
    yt = get_yt(config=config)
    return yt.get_attribute(
        table_path,
        attribute_name,
        False
    )


@retry(stop_max_attempt_number=3, wait_fixed=5000, retry_on_exception=(YtIncorrectResponse,))
def set_table_attribute(config, table_path, attribute_name, attribute_value):
    yt = get_yt(config=config)
    yt.set_attribute(
        table_path,
        attribute_name,
        attribute_value,
    )
