import functools
import logging
from typing import Callable

from django.http import (
    HttpRequest,
    HttpResponse,
)
from django.shortcuts import render
from django_yauth.user import YandexUser
from passport.backend.py_adm.core.tvm.tvm import get_tvmauth_credentials_manager


log = logging.getLogger('py_adm.roles')

TDjangoViewCallable = Callable[[HttpRequest, ...], HttpResponse]
TDjangoViewDecorator = Callable[[TDjangoViewCallable], TDjangoViewCallable]


def get_missing_roles(request: HttpRequest, roles: list[str]) -> list[str]:
    yauser: YandexUser = getattr(request, 'yauser', None)
    if yauser is None:
        raise RuntimeError('No request.yauser field')
    manager = get_tvmauth_credentials_manager()
    # Здесь используется raw_user_ticket, потому что авторизация по куке
    # .user_ticket при ней не устанавливается
    return [
        role for role in roles
        if not manager.check_user_role(
            yauser=yauser,
            uid=yauser.uid,
            role=role,
        )
    ]


def get_user_roles(request: HttpRequest):
    yauser: YandexUser = getattr(request, 'yauser', None)
    if yauser is None:
        raise RuntimeError('No request.yauser field')
    manager = get_tvmauth_credentials_manager()
    return manager.get_user_roles(yauser=yauser, uid=yauser.uid)


def require_roles(roles: list[str]) -> TDjangoViewDecorator:
    def decorator(outer: TDjangoViewCallable) -> TDjangoViewCallable:
        @functools.wraps(outer)
        def inner(request: HttpRequest, *args, **kwargs) -> HttpResponse:
            missing_roles = get_missing_roles(request, roles)
            if missing_roles:
                log.debug(
                    'Missing roles: {}, existing roles: {}'.format(
                        missing_roles, get_user_roles(request),
                    ),
                )
                return render(
                    request,
                    'not_enough_roles.html',
                    context=dict(roles=missing_roles),
                    status=403,
                )
            else:
                return outer(request, *args, **kwargs)

        return inner

    return decorator
