import logging

from django.conf import settings
from django_yauth.user import YandexUser
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
import tvmauth


log = logging.getLogger('py_adm.tvm')


@lazy_loadable()
class TvmAuthCredentialsManager:
    def __init__(self):
        tvm_settings = tvmauth.TvmApiClientSettings(
            self_tvm_id=settings.TVM_ID,
            self_secret=settings.TVM_SECRET,
            dsts=settings.TVM_DESTINATIONS,
            disk_cache_dir=settings.TVM_CACHE_DIR,
            enable_user_ticket_checking=True,
            tirole_host=settings.TIROLE_HOST,
            tirole_port=settings.TIROLE_PORT,
            tirole_tvmid=settings.TIROLE_TVMID,
            fetch_roles_for_idm_system_slug=settings.IDM_SYSTEM_SLUG,
        )
        self.client = tvmauth.TvmClient(tvm_settings)

    def get_ticket_by_alias(self, alias: str):
        return self.client.get_service_ticket_for(alias=alias)

    def check_user_role(self, yauser: YandexUser, uid: int, role: str) -> bool:
        checked_ticket = self.client.check_user_ticket(
            ticket=yauser.raw_user_ticket,
            overrided_bb_env=tvmauth.BlackboxEnv.ProdYateam,
        )
        if uid not in checked_ticket.uids:
            log.debug('Uid {} not in user ticket'.format(uid))
            return False

        roles = self.client.get_roles()
        if not roles.check_user_role(checked_ticket, role, selected_uid=uid):
            log.debug('Uid {} doesn\'t have role {}'.format(uid, role))
            return False

        return True

    def get_user_roles(self, yauser: YandexUser, uid: int):
        checked_ticket = self.client.check_user_ticket(
            ticket=yauser.raw_user_ticket,
            overrided_bb_env=tvmauth.BlackboxEnv.ProdYateam,
        )
        if uid not in checked_ticket.uids:
            log.debug('Uid {} not in user ticket'.format(uid))
            return False

        roles = self.client.get_roles()
        return roles.get_user_roles(checked_ticket, uid)


def get_tvmauth_credentials_manager() -> TvmAuthCredentialsManager:
    manager = LazyLoader.get_instance('TvmAuthCredentialsManager')
    return manager
