import datetime
from typing import Any

from django.http import (
    HttpRequest,
    HttpResponse,
)
from django.shortcuts import render
from django_yauth.decorators import yalogin_required
from passport.backend.py_adm.core.common.historydb import get_historydb_api
from passport.backend.py_adm.core.common.roles import require_roles
from passport.backend.py_adm.pushes.forms import (
    PushForm,
    PushHistoryForm,
)
from passport.backend.utils.time import datetime_to_unixtime

ROLE_VIEW_ADMIN = '/role/user/access_type/view/'


def datetime_to_unixtime_day_end(d: datetime.datetime):
    return datetime_to_unixtime(d.replace(hour=23, minute=59, second=59))


def make_context() -> dict[str, Any]:
    date_year_ago = datetime.datetime.now() - datetime.timedelta(days=365)
    return dict(
        default_date_from=date_year_ago.strftime('%Y-%m-%d'),
        default_date_to=datetime.datetime.now().strftime('%Y-%m-%d'),
    )


@yalogin_required
@require_roles([ROLE_VIEW_ADMIN])
def push_history_view(request: HttpRequest) -> HttpResponse:
    context = make_context()

    if request.GET.get('date_from'):
        form = PushHistoryForm(request.GET)
        if form.is_valid():
            api = get_historydb_api()
            context['history'] = api.push_history_by_fields(
                from_ts=int(datetime_to_unixtime(form.cleaned_data['date_from'])),
                to_ts=int(datetime_to_unixtime_day_end(form.cleaned_data['date_to'])),
                uid=form.cleaned_data['uid'] or None,
                device=form.cleaned_data['device_id'] or None,
                app=form.cleaned_data['app'] or None,
            )
    else:
        form = PushHistoryForm()

    context['history_form'] = form

    return render(request, 'push_history.html', context)


@yalogin_required
@require_roles([ROLE_VIEW_ADMIN])
def push_view(request: HttpRequest) -> HttpResponse:
    context = make_context()

    if request.GET.get('push_id'):
        form = PushForm(request.GET)
        if form.is_valid():
            api = get_historydb_api()
            context['push_data'] = api.push_history_by_push_id(
                push=form.cleaned_data['push_id'],
            )

    context['history_form'] = PushHistoryForm()

    return render(request, 'push.html', context)
