# -*- coding: utf-8 -*-
from collections.abc import Mapping
from datetime import datetime
import logging
import random
import string
from typing import (
    Dict,
    Optional,
    Union,
)
from urllib.parse import urljoin

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.logging_utils.helpers import mask_sensitive_fields
from passport.backend.qa.autotests.base.secrets import get_tvm_client
from passport.backend.qa.autotests.base.settings.qa_proxy import (
    QA_PROXY_TARGET_URL_HEADER_NAME,
    QA_PROXY_TVM_DST_ALIAS,
    QA_PROXY_TVM_TICKET_HEADER_NAME,
    QA_PROXY_URL,
)
import pytest


DEFAULT_TIMEOUT = 3
DEFAULT_RETRIES = 3
DATETIME_FORMAT_FOR_LOG = '%d %b %Y %H:%M:%S'

log = logging.getLogger(__name__)


class BaseAutotestsBuilderError(Exception):
    pass


class AutotestsBuilderTemporaryError(BaseAutotestsBuilderError):
    pass


class AutotestsBuilderUnexpectedResponseError(BaseAutotestsBuilderError):
    pass


def _format_param_list(params, indent=2):
    if isinstance(params, Mapping):
        return '\n'.join([
            '{}{}: {}'.format(' ' * indent, param, value)
            for param, value in sorted((params or {}).items())
        ])
    else:
        return params


def _make_test_req_id(length=10):
    return ''.join(random.sample(string.ascii_letters + string.digits, length))


class BaseAutotestsBuilder(BaseBuilder, JsonParserMixin):
    base_error_class = BaseAutotestsBuilderError
    temporary_error_class = AutotestsBuilderTemporaryError
    parser_error_class = AutotestsBuilderUnexpectedResponseError

    def __init__(self, base_url, timeout=DEFAULT_TIMEOUT, retries=DEFAULT_RETRIES, tvm_dst_alias=None, use_proxy=False):
        tvm_credentials_manager = None
        if tvm_dst_alias:
            tvm_credentials_manager = get_tvm_client()
        target_url = base_url
        self.proxy_headers = {}
        if use_proxy:
            target_url = QA_PROXY_URL
            self.proxy_headers = make_proxy_headers(base_url)
        super(BaseAutotestsBuilder, self).__init__(
            url=target_url,
            timeout=timeout,
            retries=retries,
            logger=log,
            tvm_dst_alias=tvm_dst_alias,
            tvm_credentials_manager=tvm_credentials_manager,
        )

    @staticmethod
    def make_error_handler(expected_http_status):
        def error_handler(response):
            pytest.allure.attach(
                f'Response ({response.status_code})',
                response.content,
                pytest.allure.attach_type.TEXT,
            )
            if response.status_code != expected_http_status:
                raise AutotestsBuilderUnexpectedResponseError(response.content)

        return error_handler

    def error_detector(self, response, raw_response):
        # может перегружаться в потомках
        pass

    def parse(self, raw_response):
        raise NotImplementedError()

    def request(self, method, path, query_params=None, form_params=None, headers=None, cookies=None, files=None,
                expected_http_status=200, **kwargs):
        query_params = dict(query_params or {}, test_req_id=_make_test_req_id())
        headers = dict(headers or {}, **self.proxy_headers)

        with pytest.allure.step('{} {}'.format(
            method,
            urljoin(self.url, path),
        )):
            pytest.allure.attach(
                'Request',
                '\n\n'.join(
                    [
                        'Time: {}'.format(datetime.now().strftime(DATETIME_FORMAT_FOR_LOG)),
                    ] + [
                        '{}:\n{}'.format(section, _format_param_list(mask_sensitive_fields(params)))
                        for section, params in {
                            'Query params': query_params,
                            'Form params': form_params,
                            'Headers': headers,
                        }.items()
                        if params
                    ],
                ),
                pytest.allure.attach_type.TEXT,
            )
            return self._request_with_retries_simple(
                url_suffix=path,
                method=method,
                params=query_params,
                data=form_params,
                headers=headers,
                cookies=cookies,
                files=files,
                http_error_handler=self.make_error_handler(expected_http_status),
                parser=self.parse,
                error_detector=self.error_detector,
                **kwargs,
            )

    def get(self, path, query_params=None, form_params=None, headers=None, cookies=None, files=None,
            expected_http_status=200):
        return self.request(
            method='GET',
            path=path,
            query_params=query_params,
            form_params=form_params,
            headers=headers,
            cookies=cookies,
            files=files,
            expected_http_status=expected_http_status,
        )

    def post(self,
             path: str,
             query_params: Optional[Dict[str, str]] = None,
             form_params: Union[Optional[Dict[str, str]], str] = None,
             headers: Optional[Dict[str, str]] = None,
             cookies: Optional[Dict[str, str]] = None,
             files=None,
             expected_http_status: int = 200):
        return self.request(
            method='POST',
            path=path,
            query_params=query_params,
            form_params=form_params,
            headers=headers,
            cookies=cookies,
            files=files,
            expected_http_status=expected_http_status,
        )

    def delete(self,
               path: str,
               query_params: Optional[Dict[str, str]] = None,
               form_params: Union[Optional[Dict[str, str]], str] = None,
               headers: Optional[Dict[str, str]] = None,
               cookies: Optional[Dict[str, str]] = None,
               files=None,
               expected_http_status: int = 200):
        return self.request(
            method='DELETE',
            path=path,
            query_params=query_params,
            form_params=form_params,
            headers=headers,
            cookies=cookies,
            files=files,
            expected_http_status=expected_http_status,
        )


class AutotestsSimpleBuilder(BaseAutotestsBuilder):
    def parse(self, raw_response):
        return raw_response


class AutotestsPlaintextBuilder(BaseAutotestsBuilder):
    def parse(self, raw_response):
        return raw_response.content.decode()


class AutotestsJsonBuilder(BaseAutotestsBuilder, JsonParserMixin):
    def parse(self, raw_response):
        self.raw_response = raw_response
        return self.parse_json(raw_response)


def make_proxy_headers(proxy_target_url):
    return {
        QA_PROXY_TVM_TICKET_HEADER_NAME: get_tvm_client().get_ticket_by_alias(QA_PROXY_TVM_DST_ALIAS),
        QA_PROXY_TARGET_URL_HEADER_NAME: proxy_target_url,
    }
