from functools import wraps

from passport.backend.qa.autotests.base.helpers.common import build_api_basic_headers
from passport.backend.qa.autotests.base.secrets import secrets
from passport.backend.qa.autotests.base.steps.cookie_steps import (
    get_cookies,
    get_cookies_by_track,
)
from passport.backend.qa.autotests.base.steps.registration_steps import register_portal_account
from passport.backend.qa.autotests.base.steps.token_steps import (
    issue_token,
    issue_token_by_sessionid,
)
from passport.backend.qa.autotests.base.steps.tracks import create_track_for_account
from passport.backend.utils.common import universal_decorator


def obtain_account_credentials(account, need_cookies=True, need_xtoken=False, allow_challenge=False):
    if need_cookies:
        account.cookies = get_cookies(account, allow_challenge=allow_challenge)
    if need_xtoken:
        account.token = issue_token(
            user=account,
            client_id=secrets.OAUTH_XTOKEN_CLIENT_ID,
            client_secret=secrets.OAUTH_XTOKEN_CLIENT_SECRET,
        )


def obtain_account_credentials_by_track(account, track_id):
    # TODO: прочитать трек: если в нём не разрешено выписывать куки,
    #  но разрешено выписывать токен - нужно инвертировать процесс.
    account.cookies = get_cookies_by_track(track_id=track_id)
    account.token = issue_token_by_sessionid(
        headers=build_api_basic_headers(cookie=account.cookies),
        client_id=secrets.OAUTH_XTOKEN_CLIENT_ID,
        client_secret=secrets.OAUTH_XTOKEN_CLIENT_SECRET,
    )


@universal_decorator
def with_portal_account(
    need_xtoken=False,
    allow_challenge=False,
    **register_kwargs
):
    def decorator(outer):
        @wraps(outer)
        def inner(*args, **kwargs):
            with register_portal_account(**register_kwargs) as account:
                obtain_account_credentials(
                    account,
                    need_xtoken=need_xtoken,
                    allow_challenge=allow_challenge,
                )
                return outer(*args, account=account, **kwargs)

        return inner

    return decorator


@universal_decorator
def with_portal_account_and_track(need_xtoken=False, register_kwargs=None):
    register_kwargs = register_kwargs or {}
    def decorator(outer):
        @wraps(outer)
        def inner(*args, **kwargs):
            with register_portal_account(**register_kwargs) as account:
                obtain_account_credentials(account, need_xtoken=need_xtoken)
                with create_track_for_account(account) as track_id:
                    return outer(*args, account=account, track_id=track_id, **kwargs)

        return inner

    return decorator
