# -*- coding: utf-8 -*-
import allure
from hamcrest.core.base_matcher import BaseMatcher
from passport.backend.qa.autotests.base.builders.proxied.blackbox import Blackbox
from passport.backend.qa.autotests.base.test_env import test_env


@allure.step('Проверка токена в ЧЯ')
def verify_token(access_token, **kwargs):
    return Blackbox().get(
        path='/blackbox',
        query_params={
            'method': 'oauth',
            'userip': test_env.user_ip,
            'oauth_token': access_token,
            'format': 'json',
            **kwargs,
        },
    )


class _TokenIsValid(BaseMatcher):
    def _matches(self, bb_response):
        return bb_response.get('status', {}).get('value') == 'VALID'

    def describe_to(self, description):
        description.append_text('Токен должен быть валидным')

    def describe_mismatch(self, bb_response, mismatch_description):
        mismatch_description.append_text('was <status={}>'.format(bb_response.get('status')))


class _TokenIsForUid(BaseMatcher):
    def __init__(self, expected_uid):
        self.expected_uid = expected_uid

    def _matches(self, bb_response):
        return bb_response.get('uid', {}).get('value') == str(self.expected_uid)

    def describe_to(self, description):
        description.append_text(f'Токен должен быть выдан аккаунтом uid={self.expected_uid}')

    def describe_mismatch(self, bb_response, mismatch_description):
        mismatch_description.append_text('was <uid={}>'.format(bb_response.get('uid')))


class _TokenIs2Legged(BaseMatcher):
    def _matches(self, bb_response):
        return 'uid' not in bb_response

    def describe_to(self, description):
        description.append_text('Токен не должен содержать информацию о пользователе')

    def describe_mismatch(self, bb_response, mismatch_description):
        mismatch_description.append_text('was <uid={}>'.format(bb_response.get('uid')))


class _TokenIsForClient(BaseMatcher):
    def __init__(self, expected_client_id):
        self.expected_client_id = expected_client_id

    def _matches(self, bb_response):
        return (
            bb_response.get('oauth', {}).get('client_id') == str(self.expected_client_id)
        )

    def describe_to(self, description):
        description.append_text(f'Токен должен быть выдан приложению client_id={self.expected_client_id}')

    def describe_mismatch(self, bb_response, mismatch_description):
        mismatch_description.append_text('was <client_id={}>'.format(bb_response.get('oauth', {}).get('client_id')))


class _TokenIsTrusted(BaseMatcher):
    def _matches(self, bb_response):
        return bool(bb_response.get('oauth', {}).get('is_xtoken_trusted'))

    def describe_to(self, description):
        description.append_text('Токен должен быть доверенным')

    def describe_mismatch(self, bb_response, mismatch_description):
        mismatch_description.append_text('was <is_xtoken_trusted={}>'.format(bb_response.get('oauth', {}).get('is_xtoken_trusted')))


def token_is_valid():
    return _TokenIsValid()


def token_is_for_user(user):
    return _TokenIsForUid(expected_uid=user.uid)


def token_is_2legged():
    return _TokenIs2Legged()


def token_is_for_client(client_id):
    return _TokenIsForClient(expected_client_id=client_id)


def token_is_trusted():
    return _TokenIsTrusted()
