from datetime import timedelta

import allure
from hamcrest import (
    all_of,
    anything,
    assert_that,
    has_entries,
    has_key,
    is_not,
)
from passport.backend.qa.autotests.base.account import Account
from passport.backend.qa.autotests.base.builders.proxied.passport_api import PassportApi
from passport.backend.qa.autotests.base.builders.proxied.social import SocialBroker
from passport.backend.qa.autotests.base.helpers.cookies import add_cookies_to_header
from passport.backend.qa.autotests.base.settings.common import PASSPORT_HOST
from passport.backend.qa.autotests.base.settings.social import DEFAULT_SOCIAL_RETPATH
from passport.backend.qa.autotests.base.test_env import test_env
from passport.backend.utils.time import get_unixtime


class SocialAuthSubmitStep:
    def __init__(self):
        self.callback_response = None
        self.profile_userid = None
        self.track_id = None

    @classmethod
    def from_profile_userid(cls, profile_userid):
        self = SocialAuthSubmitStep()
        self.profile_userid = profile_userid

        return self

    def execute(self):
        return self()

    @allure.step('Первый шаг соц. авторизации (callback)')
    def __call__(self):
        application = 'vkontakte'
        retpath = DEFAULT_SOCIAL_RETPATH

        rv = PassportApi().post(
            form_params=dict(
                application=application,
                retpath=retpath,
            ),
            headers={
                'Ya-Client-Accept-Language': '',
                'Ya-Client-Cookie': '',
                'Ya-Client-Host': PASSPORT_HOST,
                'Ya-Client-User-Agent': test_env.user_agent,
                'Ya-Consumer-Client-Ip': test_env.user_ip,
            },
            path='/1/bundle/auth/social/start/',
        )
        assert_that(
            rv,
            has_entries(
                status='ok',
                track_id=anything(),
            ),
            'res={}'.format(rv),
        )
        self.track_id = rv.get('track_id')

        rv = SocialBroker().post(
            form_params=dict(
                application_name=application,
                profile_userid=self.profile_userid,
                profile_username=self.profile_userid,
                token_expired=str(get_unixtime() + timedelta(minutes=5).total_seconds()),
                token_scope='does-not-matter',
                token_value='does-not-matter',
            ),
            path='/brokerapi/test/create_task',
        )
        assert_that(
            rv,
            has_entries(
                status='ok',
                task_id=anything(),
            ),
            'res={}'.format(rv),
        )
        task_id = rv.get('task_id')

        rv = PassportApi().post(
            form_params=dict(
                status='ok',
                task_id=task_id,
                track_id=self.track_id,
            ),
            headers={
                'Ya-Client-Accept-Language': '',
                'Ya-Client-Host': PASSPORT_HOST,
                'Ya-Client-User-Agent': test_env.user_agent,
                'Ya-Consumer-Client-Ip': test_env.user_ip,
            },
            path='/1/bundle/auth/social/callback/',
        )

        self.callback_response = rv

        return self


class AuthMultiStep:
    default_headers = {
        'Ya-Client-Host': PASSPORT_HOST,
        'Ya-Client-User-Agent': test_env.user_agent,
        'Ya-Client-Cookie': '',
        'Ya-Consumer-Client-Ip': test_env.user_ip,
    }

    def __init__(self, account: Account):
        self.account = account

    @allure.step('Первый шаг multistep аутентификации')
    def start(self, with_2fa_pictures=False, check_result=True, with_cookie_lah=False):
        headers = dict(self.default_headers)
        if with_cookie_lah:
            headers['Ya-Client-Cookie'] = add_cookies_to_header(
                headers['Ya-Client-Cookie'],
                'lah',
                self.account.parsed_cookies['lah'],
            )
        rv = PassportApi().post(
            path='/1/bundle/auth/password/multi_step/start/',
            form_params={
                'login': self.account.login,
                'with_2fa_pictures': with_2fa_pictures,
            },
            headers=headers,
        )
        if check_result:
            assert_that(
                rv,
                has_entries(
                    status='ok',
                    can_authorize=True,
                    track_id=is_not(None),
                    csrf_token=is_not(None),
                ),
            )

        return rv

    @allure.step('Второй шаг multistep аутентификации - по коду из СМС')
    def auth_by_sms_code(self, track_id, check_result=True):
        rv = PassportApi().post(
            path='/1/bundle/auth/password/multi_step/commit_sms_code/',
            form_params={
                'track_id': track_id,
            },
            headers=self.default_headers,
        )
        if check_result:
            assert_that(
                rv,
                has_entries(
                    status='ok',
                    track_id=is_not(None),
                ),
                'rv={}'.format(rv),
            )

        return rv

    @allure.step('Второй шаг multistep аутентификации - по паролю')
    def auth_by_password(self, track_id, password, check_result=True):
        rv = PassportApi().post(
            path='/1/bundle/auth/password/multi_step/commit_password/',
            form_params={
                'track_id': track_id,
                'password': password,
            },
            headers=self.default_headers,
        )
        if check_result:
            assert_that(
                rv,
                has_entries(
                    status='ok',
                    track_id=is_not(None),
                ),
                'rv={}'.format(rv),
            )

        return rv

    @allure.step('Второй шаг multistep аутентификации - по 2fa-магии')
    def auth_by_magic(self, track_id, csrf_token, check_result=True):
        rv = PassportApi().post(
            path='/1/bundle/auth/password/multi_step/commit_magic/',
            form_params={
                'track_id': track_id,
                'csrf_token': csrf_token,
            },
            headers=self.default_headers,
        )
        if check_result:
            assert_that(
                rv,
                has_entries(
                    status='ok',
                    track_id=is_not(None),
                ),
                'rv={}'.format(rv),
            )

        return rv

    @allure.step('Передача otp в трек')
    def fill_track_with_otp(self, track_id, otp, selected_2fa_picture=None, check_response=True):
        rv = PassportApi().post(
            path='/1/bundle/auth/otp/prepare/',
            form_params=dict(
                track_id=track_id,
                login=self.account.login,
                otp=otp,
                selected_2fa_picture=selected_2fa_picture,
            )
        )
        if check_response:
            assert_that(
                rv,
                has_entries(
                    status='ok',
                ),
            )

        return rv

    def check_account_suggest_by_phone_availability(self, phone_number, check_result=True):
        rv = PassportApi().get(
            path='/1/bundle/auth/suggest/by_phone/check_availability/',
            query_params={
                'phone_number': phone_number,
            },
            headers=self.default_headers,
        )
        if check_result:
            assert_that(
                rv,
                has_key('is_suggest_available'),
                'rv={}'.format(rv),
            )
        return rv

    def get_account_suggest_by_phone(self, track_id, check_result=True):
        rv = PassportApi().get(
            path='/1/bundle/auth/suggest/by_phone/list/',
            query_params={
                'track_id': track_id,
            },
            headers=self.default_headers,
        )
        if check_result:
            assert_that(
                rv,
                all_of(
                    has_key('accounts'),
                    has_key('allowed_registration_flows'),
                ),
                'rv={}'.format(rv),
            )
        return rv

    @allure.step('Второй шаг multistep аутентификации - после саджеста по номеру телефона')
    def auth_after_suggest_by_phone(self, track_id, check_result=True):
        rv = PassportApi().post(
            path='/1/bundle/auth/password/multi_step/after_suggest_by_phone/',
            form_params={
                'track_id': track_id,
                'uid': self.account.uid,
            },
            headers=self.default_headers,
        )
        if check_result:
            assert_that(
                rv,
                has_entries(
                    status='ok',
                    track_id=is_not(None),
                ),
                'rv={}'.format(rv),
            )

        return rv
