from abc import (
    ABC,
    abstractmethod,
)
import json
from typing import Optional

import allure
from hamcrest import (
    anything,
    assert_that,
    has_entries,
    has_entry,
)
from passport.backend.qa.autotests.base.account import Account
from passport.backend.qa.autotests.base.builders.proxied.passport_api import PassportApi
from passport.backend.qa.autotests.base.settings.common import PASSPORT_HOST
from passport.backend.qa.autotests.base.steps.blackbox import userinfo_by_uid
from passport.backend.qa.autotests.base.steps.phone import (
    confirm_phone_in_track,
    get_secure_phone_from_uid,
)
from passport.backend.qa.autotests.base.steps.tracks import (
    read_track,
    set_track_state,
)
from passport.backend.qa.autotests.base.test_env import test_env


TEST_CHALLENGE_RETPATH = 'https://pay.yandex.ru'


@allure.step('Включение принудительного челленджа')
def enable_force_challenge(account):
    rv = PassportApi().post(
        form_params=dict(
            force_challenge='1',
        ),
        path=f'/2/account/{account.uid}/options/',
    )
    assert_that(rv, has_entries(status='ok'))


class BasePassChallengeStep(ABC):
    account: Optional[Account]

    def __init__(self):
        self.challenges_enabled = None
        self.default_challenge = None
        self.track_id = None
        self.account = None

    def with_track(self, track_id):
        self.track_id = track_id
        return self

    def with_account(self, account):
        self.account = account
        return self

    def execute(self):
        return self()

    @allure.step('Прохождение челленджа')
    def __call__(self):
        self.challenge_submit()
        self.challenge_commit()

    @allure.step('Получение списка доступных челленжей')
    def challenge_submit(self, check_response=True):
        rv = self._challenge_submit_implementation()

        if check_response:
            assert_that(
                rv,
                has_entries(
                    challenges_enabled=anything(),
                    default_challenge=anything(),
                    status='ok',
                ),
            )

        self.challenges_enabled = rv.get('challenges_enabled')
        self.default_challenge = rv.get('default_challenge')
        return rv

    @allure.step('Завершение прохождения челленжа')
    def challenge_commit(self, challenge=None, check_response=True):
        rv = self._challenge_commit_implementation(challenge or self.default_challenge)
        if check_response:
            assert_that(
                rv,
                has_entries(status='ok'),
            )
        return rv

    @abstractmethod
    def _challenge_submit_implementation(self):
        pass

    @abstractmethod
    def _challenge_commit_implementation(self, challenge):
        pass

    def get_account_from_track(self):
        rv = read_track(self.track_id)
        assert_that(
            rv,
            has_entry('uid', anything()),
        )
        self.account = Account(uid=rv['uid'])

    def get_challenge_answer(self, challenges_enabled, challenge):
        if challenge == 'phone':
            return self.get_secure_phone_number()
        elif challenge == 'phone_confirmation':
            confirm_phone_in_track(
                track_id=self.track_id,
                phone_id=challenges_enabled[challenge]['phone_id'],
            )
            return
        elif challenge == '3ds':
            set_track_state(
                track_id=self.track_id,
                payment_status='authorized',
            )
            return
        elif challenge == 'email':
            return self.get_account_email_addresses()[0]
        else:
            raise NotImplementedError(f'Cannot pass challenge `{challenge}` automatically')

    def get_secure_phone_number(self):
        if self.account is None:
            self.get_account_from_track()

        return get_secure_phone_from_uid(self.account.uid)

    def get_account_email_addresses(self):
        if self.account is None:
            self.get_account_from_track()

        userinfo = userinfo_by_uid(self.account.uid, getemails='all', email_attributes='1')
        return [e['attributes']['1'] for e in userinfo['users'][0]['emails']]


class PassAuthChallengeStep(BasePassChallengeStep):
    default_answer = None

    def _challenge_submit_implementation(self):
        return PassportApi().post(
            form_params=dict(
                track_id=self.track_id,
            ),
            path='/1/bundle/auth/password/challenge/submit/',
        )

    def _challenge_commit_implementation(self, challenge):
        return PassportApi().post(
            form_params=dict(
                answer=str(self.default_answer or self.get_challenge_answer(
                    challenges_enabled=self.challenges_enabled,
                    challenge=challenge,
                )),
                challenge=challenge,
                track_id=self.track_id,
            ),
            path='/1/bundle/auth/password/challenge/commit/',
        )

    def set_track_state(self, antifraud_tags=None):
        set_track_state(
            self.track_id,
            is_auth_challenge_shown=True,
            antifraud_tags=antifraud_tags,
        )


class PassStandaloneChallengeStep(BasePassChallengeStep):
    def _make_headers(self, cookies=None):
        return {
            'Ya-Client-Host': PASSPORT_HOST,
            'Ya-Client-User-Agent': test_env.user_agent,
            'Ya-Client-Cookie': cookies or '',
            'Ya-Consumer-Client-Ip': test_env.user_ip,
        }

    @allure.step('Создание трека для chaas')
    def challenge_create_track(self, retpath=TEST_CHALLENGE_RETPATH, antifraud_tags=None, **kwargs):
        rv = PassportApi().post(
            form_params=dict(
                uid=self.account.uid,
                retpath=retpath,
                antifraud_tags=json.dumps(antifraud_tags) if antifraud_tags else None,
                **kwargs,
            ),
            path='/1/bundle/challenge/standalone/create_track/',
        )
        assert_that(
            rv,
            has_entries(
                status='ok',
                track_id=anything(),
            ),
        )
        return rv['track_id']

    def _challenge_submit_implementation(self):
        return PassportApi().post(
            form_params=dict(
                track_id=self.track_id,
            ),
            headers=self._make_headers(self.account.cookies),
            path='/1/bundle/challenge/standalone/submit/',
        )

    def _challenge_commit_implementation(self, challenge):
        return PassportApi().post(
            form_params=dict(
                answer=str(self.get_challenge_answer(
                    challenges_enabled=self.challenges_enabled,
                    challenge=challenge,
                )),
                challenge=challenge,
                track_id=self.track_id,
            ),
            headers=self._make_headers(self.account.cookies),
            path='/1/bundle/challenge/standalone/commit/',
        )
