from contextlib import contextmanager
from functools import wraps
from typing import (
    Callable,
    ContextManager,
    Dict,
)

import allure
from hamcrest import (
    assert_that,
    has_key,
)
from passport.backend.qa.autotests.base.builders.proxied.passport_api import PassportApi
from passport.backend.qa.autotests.base.domain import Domain
from passport.backend.qa.autotests.base.settings import ENV
from passport.backend.qa.autotests.base.settings.common import PASSPORT_HOST
from passport.backend.qa.autotests.base.test_env import test_env


class DomainAlreadyExists(Exception):
    pass


def _domain_context_manager(domain_func) -> Callable[..., ContextManager[Domain]]:
    @contextmanager
    @wraps(domain_func)
    def inner(**kwargs) -> ContextManager[Domain]:
        with allure.step('Создание тестового домена'):
            # Технически домен можно создать в ятиме, но практического смысла это не имеет.
            # Возможность создать домен - сайдэффект того, что используется один и тот же бекенд passport-api
            # в ятиме и в проде.
            # За бессмысленностью данной операции она запрещена.
            if ENV in ('intranet_testing', 'intranet_rc', 'intranet_production'):
                raise RuntimeError('Регистрация домена в ятиме не имеет смысла')
            domain = domain_func(**kwargs)
        try:
            yield domain
        finally:
            with allure.step('Удаление тестового домена'):
                _delete_domain(domain)
    return inner


def _delete_domain(domain):
    rv = PassportApi().delete(f'/1/bundle/pdd/domain/{domain.domain_id}/')
    assert rv.get('status') == 'ok' or 'domain.not_found' in rv.get('errors', [])


def _assert_domain_created(rv: Dict):
    if rv.get('status') == 'ok':
        return

    errors = rv.get('errors', [])
    if 'domain.already_exists' in errors:
        raise DomainAlreadyExists()

    raise RuntimeError('Не удалось создать домен: {}'.format(''.join(errors)))


@_domain_context_manager
def add_domain(admin_uid: int, domain: str, check_result=True):
    form_params = {
        'domain': domain,
        'admin_uid': admin_uid,
    }
    rv = PassportApi().post(
        headers={
            'Ya-Client-Accept-Language': '',
            'Ya-Client-User-Agent': test_env.user_agent,
            'Ya-Consumer-Client-Ip': test_env.user_ip,
            'Ya-Client-Host': PASSPORT_HOST,
        },
        path='/1/bundle/pdd/domain/',
        form_params=form_params,
    )
    if check_result:
        _assert_domain_created(rv)
        assert_that(rv, has_key('domain_id'))
        assert_that(rv, has_key('domain'))
    return Domain(
        domain=rv['domain'],
        domain_id=rv['domain_id'],
    )
