import base64
from hashlib import sha256
import subprocess
from time import time

import allure
from hamcrest import (
    assert_that,
    has_entries,
    is_not,
)
from passport.backend.qa.autotests.base.builders.proxied.passport_api import PassportApi
from passport.backend.qa.autotests.base.helpers.cookies import (
    dump_cookies_to_header,
    parse_cookies,
)
from passport.backend.qa.autotests.base.settings.common import PASSPORT_HOST
from passport.backend.qa.autotests.base.steps.phone import confirm_phone_in_track
from passport.backend.qa.autotests.base.test_env import test_env
import yatest.common as yc


class TotpGenerator:
    TOTP_SECRET_LENGTH = 16  # 128 бит

    def __init__(self, b32_app_secret, pin):
        self.b64_secret = self._make_b64_secret(b32_app_secret, pin)

    def _make_b64_secret(self, b32_app_secret, pin):
        if len(b32_app_secret) % 8 != 0:
            # восстановим паддинг, если требуется
            b32_app_secret += '=' * (8 - len(b32_app_secret) % 8)
        app_secret = base64.b32decode(b32_app_secret)

        if len(app_secret) > self.TOTP_SECRET_LENGTH:
            raise ValueError('Wrong secret bytes count: %s' % len(app_secret))
        app_secret = chr(0).encode() * (self.TOTP_SECRET_LENGTH - len(app_secret)) + app_secret

        secret = str(pin).encode() + app_secret
        return base64.b64encode(
            sha256(secret).digest(),
        ).decode().strip('=')

    def make_otp(self):
        gentotp_path = yc.build_path() + '/passport/infra/daemons/blackbox/tools/gentotp/gentotp'
        timestamp = int(time())
        password_length = 8
        password_type = 'Letters'
        return subprocess.check_output([
            gentotp_path,
            str(password_length),
            self.b64_secret,
            str(timestamp),
            password_type,
        ]).strip()


class EnableOtpStep:
    def __init__(self):
        self.track_id = None
        self.account = None
        self.app_secret = None
        self.pin = None
        self.push_setup_secret = None

    def with_account(self, account):
        self.account = account
        return self

    def execute(self, **kwargs):
        return self(**kwargs)

    @allure.step('Включение 2фа')
    def __call__(self, custom_pin=None, device_id=None):
        rv = self.submit()
        confirm_phone_in_track(
            track_id=self.track_id,
            phone_number=rv['secure_number']['e164'],
        )
        self.get_secret()
        if custom_pin is not None:
            self.set_pin(pin=custom_pin)
        otp = self.make_otp()
        self.check_otp(otp=otp)
        if device_id is not None:
            self.save_device(device_id)
        self.commit()
        # Поправим время последнего ввода otp, чтобы последующий код не получал ошибку, что такой otp уже проверялся
        self.set_otp_check_time()

    @staticmethod
    def _make_headers(cookies):
        return {
            'Ya-Client-Host': PASSPORT_HOST,
            'Ya-Client-User-Agent': test_env.user_agent,
            'Ya-Client-Cookie': cookies,
            'Ya-Consumer-Client-Ip': test_env.user_ip,
        }

    @allure.step('Генерация одноразового пароля')
    def make_otp(self):
        return TotpGenerator(b32_app_secret=self.app_secret, pin=self.pin).make_otp()

    @allure.step('Подмена времени предыдущей проверки одноразового пароля')
    def set_otp_check_time(self, otp_check_time=None, check_response=True):
        if otp_check_time is None:
            # такое значение заведомо позволит проверить одноразовый пароль ещё раз
            otp_check_time = int(time()) - 60

        rv = PassportApi().post(
            path='/1/bundle/otp/set_check_time/',
            form_params=dict(
                uid=self.account.uid,
                totp_check_time=otp_check_time,
            )
        )
        if check_response:
            assert_that(
                rv,
                has_entries(
                    status='ok',
                ),
            )

        return rv

    @allure.step('Начало процесса включения 2фа')
    def submit(self, check_response=True):
        rv = PassportApi().post(
            path='/2/bundle/otp/enable/submit/',
            headers=self._make_headers(self.account.cookies)
        )
        if check_response:
            assert_that(
                rv,
                has_entries(
                    status='ok',
                    track_id=is_not(None),
                    secure_number=has_entries(
                        e164=is_not(None),
                    ),
                ),
            )

        self.track_id = rv.get('track_id')
        return rv

    @allure.step('Генерация 2фа-секрета')
    def get_secret(self, check_response=True):
        rv = PassportApi().post(
            path='/1/bundle/otp/enable/get_secret/',
            form_params=dict(
                track_id=self.track_id,
            ),
            headers=self._make_headers(self.account.cookies)
        )
        if check_response:
            assert_that(
                rv,
                has_entries(
                    status='ok',
                    track_id=is_not(None),
                    pin=is_not(None),
                    app_secret=is_not(None),
                    push_setup_secret=is_not(None),
                ),
            )

        self.app_secret = rv.get('app_secret')
        self.pin = rv.get('pin')
        self.push_setup_secret = rv.get('push_setup_secret')
        return rv

    @allure.step('Выбор пина')
    def set_pin(self, pin, check_response=True):
        rv = PassportApi().post(
            path='/1/bundle/otp/enable/set_pin/',
            form_params=dict(
                track_id=self.track_id,
                pin=pin,
            ),
            headers=self._make_headers(self.account.cookies)
        )
        if check_response:
            assert_that(
                rv,
                has_entries(
                    status='ok',
                    track_id=is_not(None),
                ),
            )

        self.pin = pin
        return rv

    @allure.step('Проверка одноразового пароля')
    def check_otp(self, otp, check_response=True):
        rv = PassportApi().post(
            path='/1/bundle/otp/enable/check_otp/',
            form_params=dict(
                track_id=self.track_id,
                otp=otp,
            ),
            headers=self._make_headers(self.account.cookies)
        )
        if check_response:
            assert_that(
                rv,
                has_entries(
                    status='ok',
                    track_id=is_not(None),
                ),
            )

        return rv

    @allure.step('Привязка устройства для получения пушей с картинками')
    def save_device(self, device_id, check_response=True):
        rv = PassportApi().post(
            path='/1/bundle/otp/enable/save_device/',
            form_params=dict(
                track_id=self.track_id,
                device_id=device_id,
                push_setup_secret=self.push_setup_secret,
            ),
            headers=self._make_headers(self.account.cookies)
        )
        if check_response:
            assert_that(
                rv,
                has_entries(
                    status='ok',
                    track_id=is_not(None),
                ),
            )

        return rv

    @allure.step('Завершение включения 2фа')
    def commit(self, check_response=True):
        rv = PassportApi().post(
            path='/1/bundle/otp/enable/commit/',
            form_params=dict(
                track_id=self.track_id,
            ),
            headers=self._make_headers(self.account.cookies)
        )
        if check_response:
            assert_that(
                rv,
                has_entries(
                    status='ok',
                    cookies=is_not(None),
                ),
            )

        self.account.password = None
        # Включение 2фа глогаутит пользователя, поэтому обновим куки на модельке аккаунта
        self.account.cookies = dump_cookies_to_header(parse_cookies(rv['cookies']))
        return rv
