import allure
from hamcrest import (
    anything,
    assert_that,
    has_entries,
    has_item,
    has_key,
)
from passport.backend.core.serializers.eav.base.base import EavSerializer
from passport.backend.qa.autotests.base.account import Account
from passport.backend.qa.autotests.base.builders.proxied.blackbox import Blackbox
from passport.backend.qa.autotests.base.builders.proxied.passport_api import PassportApi
from passport.backend.qa.autotests.base.settings.common import PASSPORT_HOST
from passport.backend.qa.autotests.base.steps.tracks import read_track
from passport.backend.qa.autotests.base.test_env import test_env


def get_confirmation_code_from_track(track_id):
    rv = read_track(track_id)
    assert_that(
        rv,
        has_key('phone_confirmation_code'),
    )
    return rv.get('phone_confirmation_code')


@allure.step('Подтверждение телефона в треке')
def confirm_phone_in_track(
    track_id: str,
    phone_number: str = None,
    phone_id: str = None,
):
    form_params=dict(
        display_language='ru',
        track_id=track_id,
    )
    if phone_id:
        form_params.update(phone_id=phone_id)
    elif phone_number:
        form_params.update(number=phone_number)
    else:
        raise ValueError('Either phone_id or phone_number is required')

    rv = PassportApi().post(
        form_params=form_params,
        headers={
            'Ya-Client-Host': PASSPORT_HOST,
            'Ya-Client-User-Agent': test_env.user_agent,
            'Ya-Consumer-Client-Ip': test_env.user_ip,
        },
        path='/1/bundle/phone/confirm/submit/',
    )
    assert_that(
        rv,
        has_entries(
            status='ok',
            track_id=anything(),
        ),
    )
    track_id = rv.get('track_id')

    confirmation_code = get_confirmation_code_from_track(track_id)

    rv = PassportApi().post(
        form_params=dict(
            code=confirmation_code,
            track_id=track_id,
        ),
        headers={
            'Ya-Client-Host': PASSPORT_HOST,
            'Ya-Client-User-Agent': test_env.user_agent,
            'Ya-Consumer-Client-Ip': test_env.user_ip,
        },
        path='/1/bundle/phone/confirm/commit/',
    )
    assert_that(
        rv,
        has_entries(
            status='ok',
        ),
        'res={}'.format(rv),
    )


@allure.step('Привязывание основного телефона')
def bind_secure_phone(
    account: Account,
    phone_number: str,
):
    rv = PassportApi().post(
        form_params=dict(
            display_language='ru',
            number=phone_number,
        ),
        headers={
            'Ya-Client-Cookie': account.cookies,
            'Ya-Client-Host': PASSPORT_HOST,
            'Ya-Client-User-Agent': test_env.user_agent,
            'Ya-Consumer-Client-Ip': test_env.user_ip,
        },
        path='/2/bundle/phone/confirm_and_bind_secure/submit/',
    )
    assert_that(
        rv,
        has_entries(
            status='ok',
            track_id=anything(),
        ),
        'res={}'.format(rv),
    )
    track_id = rv.get('track_id')

    confirmation_code = get_confirmation_code_from_track(track_id)

    rv = PassportApi().post(
        form_params=dict(
            code=confirmation_code,
            track_id=track_id,
        ),
        headers={
            'Ya-Client-Cookie': account.cookies,
            'Ya-Client-Host': PASSPORT_HOST,
            'Ya-Client-User-Agent': test_env.user_agent,
            'Ya-Consumer-Client-Ip': test_env.user_ip,
        },
        path='/2/bundle/phone/confirm_and_bind_secure/commit/',
    )
    assert_that(
        rv,
        has_entries(
            status='ok',
        ),
        'res={}'.format(rv),
    )


@allure.step('Привязывание основного телефона и алиасификация')
def bind_and_aliasify_secure_phone(
    account: Account,
    phone_number: str,
):
    rv = PassportApi().post(
        form_params=dict(
            display_language='ru',
            number=phone_number,
            uid=account.uid,
        ),
        headers={
            'Ya-Client-Cookie': account.cookies,
            'Ya-Client-Host': PASSPORT_HOST,
            'Ya-Client-User-Agent': test_env.user_agent,
            'Ya-Consumer-Client-Ip': test_env.user_ip,
        },
        path='/1/bundle/phone/confirm_and_bind_secure_and_aliasify/submit/',
    )
    assert_that(
        rv,
        has_entries(
            status='ok',
            track_id=anything(),
        ),
        'res={}'.format(rv),
    )
    track_id = rv.get('track_id')

    confirmation_code = get_confirmation_code_from_track(track_id)

    rv = PassportApi().post(
        form_params=dict(
            code=confirmation_code,
            track_id=track_id,
            uid=account.uid,
            password=account.password,
        ),
        headers={
            'Ya-Client-Cookie': account.cookies,
            'Ya-Client-Host': PASSPORT_HOST,
            'Ya-Client-User-Agent': test_env.user_agent,
            'Ya-Consumer-Client-Ip': test_env.user_ip,
        },
        path='/1/bundle/phone/confirm_and_bind_secure_and_aliasify/commit/',
    )
    assert_that(
        rv,
        has_entries(
            status='ok',
        ),
        'res={}'.format(rv),
    )


def get_secure_phone(account: Account):
    return get_secure_phone_from_uid(account.uid)


@allure.step('Выяснение номера основного телефона')
def get_secure_phone_from_uid(uid: int):
    rv = Blackbox().post(
        '/blackbox',
        form_params=dict(
            attributes=EavSerializer.attr_name_to_type('phones.secure'),
            format='json',
            getphones='bound',
            method='userinfo',
            phone_attributes=EavSerializer.ext_attr_name_to_type('phone', 'number'),
            uid=str(uid),
            userip=test_env.user_ip,
        ),
    )
    assert_that(
        rv,
        has_entries(
            users=has_item(
                has_entries(id=str(uid)),
            ),
        ),
        'res={}'.format(rv),
    )
    user = ([u for u in rv.get('users', list()) if u['id'] == str(uid)] or [dict()])[0]

    assert_that(
        user,
        has_entries(
            attributes=has_entries(**{
                str(EavSerializer.attr_name_to_type('phones.secure')): anything(),
            }),
        ),
    )
    secure_phone_id = user.get('attributes', dict()).get(str(EavSerializer.attr_name_to_type('phones.secure')))

    assert_that(
        user,
        has_entries(
            phones=has_item(
                has_entries(
                    attributes=has_entries(**{
                        str(EavSerializer.ext_attr_name_to_type('phone', 'number')): anything(),
                    }),
                    id=str(secure_phone_id),
                ),
            ),
        ),
        'res={}'.format(rv),
    )
    phone_attrs = ([p for p in user.get('phones', list()) if p['id'] == str(secure_phone_id)] or [dict()])[0]
    phone_number = (
        phone_attrs
        .get('attributes', dict())
        .get(str(EavSerializer.ext_attr_name_to_type('phone', 'number')))
    )
    return '+' + phone_number
