from contextlib import contextmanager
from typing import (
    ContextManager,
    Optional,
)

import allure
from hamcrest import (
    assert_that,
    has_entries,
    has_entry,
    has_item,
)
from passport.backend.core.crypto.utils import dumb_hash_string
from passport.backend.qa.autotests.base.account import Account
from passport.backend.qa.autotests.base.builders.proxied.passport_api import PassportApi
from passport.backend.qa.autotests.base.builders.proxied.push_api import (
    PushApi,
    WsSession,
)
from passport.backend.qa.autotests.base.helpers.common import retry_codeblock
from passport.backend.qa.autotests.base.settings.common import PASSPORT_HOST
from passport.backend.qa.autotests.base.test_env import test_env
from passport.backend.utils.common import noneless_dict


TEST_DEVICE_ID = 'device123'
TEST_DEVICE_TOKEN = 'Dev1Ce/T0K3n'
TEST_APP_ID_ANDROID = 'ru.yandex.auth.client'
TEST_APP_PLATFORM_ANDROID = 'Android 118 Samsung Honor 3310'
TEST_APP_ID_IOS = 'ru.yandex.mobile.auth.sample'
TEST_APP_ID_BLACKLISTED_ANDROID = 'ru.yandex.autotests.blk.pushes.android'
TEST_APP_PLATFORM_IOS = 'iPhone 1000'


class PushApiStep:
    account: Optional[Account]

    def __init__(self):
        self.account = None

    def with_account(self, account: Account):
        self.account = account
        return self

    def _make_passport_headers(self, token: str = ''):
        headers = {
            'Ya-Client-Host': PASSPORT_HOST,
            'Ya-Client-User-Agent': test_env.user_agent,
            'Ya-Consumer-Client-Ip': test_env.user_ip,
        }
        if token:
            headers['Ya-Consumer-Authorization'] = 'OAuth ' + token
        return headers

    @staticmethod
    def _generate_uuid(deviceid: str, app_id: str):
        return dumb_hash_string(deviceid + app_id)

    @allure.step('Подписка на пуши')
    def subscribe(
        self,
        app_id=TEST_APP_ID_ANDROID,
        app_platform=TEST_APP_PLATFORM_ANDROID,
        deviceid=TEST_DEVICE_ID,
        device_token=TEST_DEVICE_TOKEN,
        am_version=None,
        app_version=None,
        client=None,
        query_params=None,
        check_response=True,
    ):
        query_params = query_params or {}
        form_params = dict(
            app_id=app_id,
            app_platform=app_platform,
            deviceid=deviceid,
            device_token=device_token,
            am_version=am_version,
            app_version=app_version,
            client=client,
        )
        rv = PassportApi().post(
            headers=self._make_passport_headers(token=self.account.token),
            query_params=query_params,
            form_params=form_params,
            path='/1/bundle/push/subscribe/',
        )
        if check_response:
            assert_that(rv, has_entries(status='ok'), 'resp: {}'.format(rv))

        return rv

    @allure.step('Отписка от пушей')
    def unsubscribe(
        self,
        app_id=TEST_APP_ID_ANDROID,
        deviceid=TEST_DEVICE_ID,
        query_params=None,
        check_response=True,
    ):
        query_params = query_params or {}
        form_params = dict(
            app_id=app_id,
            deviceid=deviceid,
            uid=self.account.uid,
        )
        rv = PassportApi().post(
            headers=self._make_passport_headers(token=self.account.token),
            query_params=query_params,
            form_params=form_params,
            path='/1/bundle/push/unsubscribe/',
        )
        if check_response:
            assert_that(rv, has_entries(status='ok'), 'resp: {}'.format(rv))

        return rv

    @contextmanager
    def with_subscription(
        self,
        app_id=TEST_APP_ID_ANDROID,
        app_platform=TEST_APP_PLATFORM_ANDROID,
        deviceid=TEST_DEVICE_ID,
        device_token=TEST_DEVICE_TOKEN,
        am_version=None,
        app_version=None,
        client=None,
        query_params=None,
        wait_for_subscription=False,
        wait_retries=10,
        retry_time=0.5,
    ):
        try:
            rv = self.subscribe(
                app_id=app_id,
                app_platform=app_platform,
                deviceid=deviceid,
                device_token=device_token,
                am_version=am_version,
                app_version=app_version,
                client=client,
                query_params=query_params,
            )
            if wait_for_subscription:
                uuid = self._generate_uuid(deviceid=deviceid, app_id=app_id)
                self.wait_for_subscriptions(
                    has_item(has_entries(uuid=uuid)),
                    wait_retries=wait_retries,
                    retry_time=retry_time,
                )
            yield rv
        finally:
            self.unsubscribe(
                app_id=app_id,
                deviceid=deviceid,
                query_params=query_params
            )

    @allure.step('Отправка пуша в АМ')
    def send_am(
        self,
        push_service,
        event_name,
        title,
        body=None,
        subtitle=None,
        webview_url=None,
        require_web_auth=False,
        check_subscriptions=False,
        check_response=True,
    ):
        form_params = noneless_dict(
            uid=self.account.uid,
            push_service=push_service,
            event_name=event_name,
            title=title,
            body=body,
            subtitle=subtitle,
            webview_url=webview_url,
            require_web_auth=require_web_auth,
            check_subscriptions=check_subscriptions,
        )
        rv = PassportApi().post(
            headers=self._make_passport_headers(),
            form_params=form_params,
            path='/1/bundle/push/send/am/',
        )
        if check_response:
            assert_that(rv, has_entries(status='ok'), 'resp: {}'.format(rv))

        return rv

    def wait_for_subscriptions(
        self,
        hamcrest_seq_matcher,
        wait_retries=4,
        retry_time=0.5,
    ):
        push_api = PushApi()

        def check():
            assert_that(
                push_api.list(self.account.uid),
                hamcrest_seq_matcher,
            )

        retry_codeblock(check, retries=wait_retries, wait=retry_time)

    @contextmanager
    def with_recv_socket(
        self,
        wait_for_subscription=False,
        wait_retries=10,
        retry_time=0.5,
    ) -> ContextManager[WsSession]:
        ws = PushApi().open_ws(self.account.uid)

        with ws:
            if wait_for_subscription:
                self.wait_for_subscriptions(
                    has_item(
                        has_entry(
                            'session',
                            ws.session,
                        ),
                    ),
                    wait_retries=wait_retries,
                    retry_time=retry_time,
                )
            yield ws
