from contextlib import contextmanager
from datetime import datetime
from functools import wraps
import random
import string
from typing import (
    Callable,
    ContextManager,
    Dict,
)

import allure
from hamcrest import (
    all_of,
    assert_that,
    has_entries,
    has_key,
    is_not,
)
from passport.backend.core.builders.frodo.utils import (
    RU_LOWER_CASE,
    RU_UPPER_CASE,
)
from passport.backend.qa.autotests.base.account import Account
from passport.backend.qa.autotests.base.builders.proxied.passport_api import PassportApi
from passport.backend.qa.autotests.base.settings.common import PASSPORT_HOST
from passport.backend.qa.autotests.base.settings.env import ENV
from passport.backend.qa.autotests.base.steps.account import _get_headers
from passport.backend.qa.autotests.base.steps.auth import SocialAuthSubmitStep
from passport.backend.qa.autotests.base.steps.cookie_steps import (
    dump_cookies_to_header,
    parse_cookies,
)
from passport.backend.qa.autotests.base.test_env import test_env
from passport.backend.utils.common import noneless_dict


DEFAULT_LOGIN_PREFIX = 'yndx-passp-tests'
FORCE_CHALLENGE_LOGIN_PREFIX = 'yndx-force-challenge'
DEFAULT_PASSWORD = 'ILikeTesting!@'

MAX_LOGIN_LENGTH = 30
MAX_LOGIN_GENERATION_ATTEMPTS = 5


class LoginNotAvailableError(Exception):
    pass


def _registration_context_manager(register_func) -> Callable[..., ContextManager[Account]]:
    @contextmanager
    @wraps(register_func)
    def inner(**kwargs) -> ContextManager[Account]:
        with allure.step('Регистрация тестового аккаунта'):
            if ENV in ('intranet_testing', 'intranet_rc', 'intranet_production'):
                raise RuntimeError('Регистрация аккаунтов в ятиме невозможна')

            account = None
            for _ in range(MAX_LOGIN_GENERATION_ATTEMPTS):
                try:
                    account = register_func(**kwargs)
                    break
                except LoginNotAvailableError:
                    continue
            if account is None:
                raise RuntimeError('Не удалось создать аккаунт: нет незанятых логинов')
        try:
            yield account
        finally:
            with allure.step('Удаление тестового аккаунта'):
                _delete_account(account)

    return inner


def _delete_account(account):
    rv = PassportApi().delete(
        path=f'/1/bundle/account/{account.uid}/',
    )
    # Может не получиться удалить аккаунт из-за отставания слейва или из-за таймаута первого ретрая.
    # Считаем это некритичным для теста.
    assert rv.get('status') == 'ok' or 'account.not_found' in rv.get('errors', [])


def _generate_login(login_template, alphabet=None, random_part_len=None):
    if alphabet is None:
        alphabet = string.ascii_letters + string.digits

    sep = '-' if '-' in alphabet else alphabet[0]
    date_part = datetime.now().strftime('%m%d')
    if random_part_len is None:
        random_part_len = max(MAX_LOGIN_LENGTH - len(sep) - len(login_template) - len(date_part), 1)
    custom_part = sep.join([
        date_part,
        ''.join(random.sample(alphabet, random_part_len)),
    ])
    return login_template.format(custom_part)


def _assert_account_registered(rv: Dict):
    if rv.get('status') == 'ok':
        return

    errors = rv.get('errors', [])
    if 'login.notavailable' in errors:
        raise LoginNotAvailableError()

    raise RuntimeError('Не удалось создать аккаунт: {}'.format(''.join(errors)))


@_registration_context_manager
def register_portal_account(
    login_prefix=DEFAULT_LOGIN_PREFIX,
    password=DEFAULT_PASSWORD,
    firstname='f',
    lastname='l',
    language='ru',
    **kwargs
) -> Account:
    login = _generate_login(login_template=login_prefix + '-{}')
    additional_args = noneless_dict(
        firstname=firstname,
        lastname=lastname,
        **kwargs,
    )
    rv = PassportApi().post(
        path='/1/bundle/account/register/',
        form_params={
            'login': login,
            'password': password,
            'ignore_stoplist': True,
            'language': language,
            **additional_args,
        },
        headers={
            'Ya-Consumer-Client-Ip': test_env.user_ip,
            'Ya-Client-User-Agent': test_env.user_agent,
        },
    )
    _assert_account_registered(rv)
    return Account(uid=rv['uid'], login=login, password=password)


@_registration_context_manager
def register_lite_account(
    login_prefix=DEFAULT_LOGIN_PREFIX, domain='not-ya.ru', password=DEFAULT_PASSWORD,
    firstname='f', lastname='l',
) -> Account:
    login = _generate_login(login_template=login_prefix + '-{}@' + domain, random_part_len=8)
    rv = PassportApi().post(
        path='/1/bundle/test/register_lite/',
        form_params={
            'login': login,
            'password': password,
            'firstname': firstname,
            'lastname': lastname,
        },
        headers={
            'Ya-Consumer-Client-Ip': test_env.user_ip,
        },
    )
    _assert_account_registered(rv)
    return Account(uid=rv['uid'], login=login, password=password)


@_registration_context_manager
def register_social_account(
    application='vkontakte',
    firstname='f',
    lastname='l',
) -> Account:
    profile_userid = _generate_login(login_template=DEFAULT_LOGIN_PREFIX + '-{}')

    social_auth_submit = SocialAuthSubmitStep.from_profile_userid(profile_userid)
    social_auth_submit()
    assert_that(
        social_auth_submit.callback_response,
        has_entries(status='ok'),
        'res={}'.format(social_auth_submit.callback_response),
    )

    if social_auth_submit.callback_response.get('state') != 'register':
        raise LoginNotAvailableError()

    rv = PassportApi().post(
        form_params=dict(
            eula_accepted='1',
            firstname=firstname,
            lastname=lastname,
            track_id=social_auth_submit.track_id,
        ),
        headers={
            'Ya-Client-Accept-Language': '',
            'Ya-Client-User-Agent': test_env.user_agent,
            'Ya-Consumer-Client-Ip': test_env.user_ip,
            'Ya-Client-Host': PASSPORT_HOST,
        },
        path='/1/bundle/auth/social/register/',
    )
    assert_that(
        rv,
        all_of(
            is_not(has_key('errors')),
            has_entries(
                account=has_key('uid'),
                status='ok',
            ),
        ),
        'res={}'.format(rv),
    )

    account = Account()
    account.uid = rv.get('account', dict()).get('uid')

    if rv.get('cookies'):
        cookies = parse_cookies(rv.get('cookies'))
        account.cookies = dump_cookies_to_header(cookies)

    return account


@_registration_context_manager
def register_scholar_account():
    login = _generate_login(
        login_template='яндкспасптесты{}',
        alphabet=RU_LOWER_CASE + RU_UPPER_CASE + string.digits,
    )
    form_params = dict(
        firstname='Владимир',
        lastname='Сидоров',
        login=login,
        password='вовочкаплюссидороваравнол',
    )
    rv = PassportApi().post(
        form_params=form_params,
        headers={
            'Ya-Consumer-Client-Ip': test_env.user_ip,
            'Ya-Client-User-Agent': test_env.user_agent,
        },
        path='/1/bundle/scholar/register/',
    )
    _assert_account_registered(rv)

    account = Account()
    attrs = [
        'firstname',
        'lastname',
        'login',
        'uid',
    ]
    for attr in attrs:
        setattr(account, attr, rv.get(attr))
    account.password = form_params['password']

    return account


@_registration_context_manager
def register_neophonish_account(
    track_id,
    firstname='f',
    lastname='l',
    **kwargs
) -> Account:
    form_params = dict(
        eula_accepted='1',
        track_id=track_id,
    )
    if firstname:
        form_params.update(firstname=firstname)
    if lastname:
        form_params.update(lastname=lastname)
    form_params.update(kwargs)
    rv = PassportApi().post(
        headers={
            'Ya-Client-Accept-Language': '',
            'Ya-Client-User-Agent': test_env.user_agent,
            'Ya-Consumer-Client-Ip': test_env.user_ip,
            'Ya-Client-Host': PASSPORT_HOST,
        },
        path='/1/bundle/account/register/neophonish/',
        form_params=form_params,
    )
    _assert_account_registered(rv)
    assert_that(rv, has_key('uid'))

    account = Account()
    account.uid = rv['uid']
    account.firstname = firstname
    account.lastname = lastname

    return account


@_registration_context_manager
def register_federal_account(login_prefix=None, **kwargs) -> Account:
    test_idp_domain = 'sso-adfs-test-domain.com'  # название домена
    login_prefix = login_prefix or 'login'
    login = _generate_login(login_template=login_prefix + '-{}@' + test_idp_domain)

    form_params = dict(
        firstname='f',
        lastname='l',
        email=login,
        domain_id=2997121,
        name_id=login,
    )
    form_params.update(kwargs)
    rv = PassportApi().post(
        headers=_get_headers(),
        path='/1/bundle/account/register/federal/',
        form_params=form_params,
    )
    _assert_account_registered(rv)
    assert_that(rv, has_key('uid'))

    account = Account()
    account.uid = rv['uid']
    account.firstname = form_params['firstname']
    account.lastname = form_params['lastname']

    return account
