from contextlib import contextmanager
import json

from hamcrest import (
    assert_that,
    has_entries,
    has_entry,
    has_key,
)
from passport.backend.qa.autotests.base.builders.proxied.passport_api import PassportApi
from passport.backend.qa.autotests.base.settings.common import PASSPORT_HOST
from passport.backend.qa.autotests.base.test_env import test_env
import pytest


def _get_headers(cookies=None):
    headers = {
        'Ya-Client-Host': PASSPORT_HOST,
        'Ya-Client-User-Agent': test_env.user_agent,
        'Ya-Consumer-Client-Ip': test_env.user_ip,
    }
    if cookies:
        headers['Ya-Client-Cookie'] = cookies

    return headers


def read_track(track_id):
    """
    Возвращает значения всех полей трека (в виде строк: типы полей игнорируются)
    """
    with pytest.allure.step('Чтение трека'):
        rv = PassportApi().get(
            path='/1/bundle/test/track/',
            query_params={'track_id': track_id},
            headers=_get_headers(),
        )
        assert_that(
            rv,
            has_entry('status', 'ok'),
        )
        return rv


def delete_track(track_id):
    with pytest.allure.step('Удаление трека'):
        rv = PassportApi().delete(
            path='/1/track/{}/'.format(track_id),
            headers=_get_headers(),
        )
        assert_that(
            rv,
            has_entry('status', 'ok'),
        )
        return rv


def set_track_state(track_id, is_auth_challenge_shown=False, antifraud_tags=None, payment_status=None):
    rv = PassportApi().post(
        form_params=dict(
            track_id=track_id,
            is_auth_challenge_shown=is_auth_challenge_shown,
            antifraud_tags=json.dumps(antifraud_tags or []),
            payment_status=payment_status,
        ),
        headers=_get_headers(),
        path='/1/bundle/test/track/',
    )
    assert_that(rv, has_entries(status='ok'))


@contextmanager
def create_track_for_account(account):
    with pytest.allure.step('Создание трека'):
        rv = PassportApi().post(
            form_params=dict(
                uid=account.uid,
            ),
            headers=_get_headers(cookies=account.cookies),
            path='/1/bundle/track/init/',
        )
        assert_that(rv, has_key('track_id'))
        try:
            yield rv['track_id']
        finally:
            delete_track(rv['track_id'])
