package main

import (
	"context"
	"flag"
	"net"
	"net/http"
	"os"
	"os/signal"
	"sync/atomic"
	"syscall"
	"time"

	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"

	"a.yandex-team.ru/passport/backend/qa_proxy/internal"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

var (
	listenAddr    string
	listenNetwork string
	logFile       string
	cfgFile       string
	healthy       int32
	throttleLimit int
	tvmAgent      *internal.TVM
)

func main() {
	flag.StringVar(&listenAddr, "listen-addr", ":8888", "server listen address or unix socket path")
	flag.StringVar(&listenNetwork, "listen-network", "tcp", "tcp, tcp4, tcp6, unix or unixpacket")
	flag.StringVar(&logFile, "log-file", "daemon.log", "log file")
	flag.StringVar(&cfgFile, "cfg-file", "proxy-config.json", "configuration file")
	flag.IntVar(&throttleLimit, "throttle", 40, "limit concurrent requests to N")
	flag.Parse()

	_ = logger.Init(logger.Config{FilePath: logFile})
	proxyCfg := internal.ReadConfig(cfgFile, logger.Log())
	tvmAgent = internal.InitTVM(proxyCfg, logger.Log())

	r := chi.NewRouter()
	r.Use(middleware.Heartbeat("/ping"))
	r.Use(middleware.RequestID)

	r.Use(internal.Throttle(throttleLimit))
	r.Use(internal.VerifyProxyAuthorization(tvmAgent, logger.Log()))
	r.Use(internal.ValidateTargetHost(proxyCfg, logger.Log()))

	r.HandleFunc("/*", internal.HandlerProxy(logger.Log()))

	server := &http.Server{
		Addr:    listenAddr,
		Handler: r,
		//ErrorLog:     logger,
		ReadTimeout:  10 * time.Second,
		WriteTimeout: 20 * time.Second,
		IdleTimeout:  10 * time.Second,
	}

	done := make(chan bool)
	quit := make(chan os.Signal, 1)
	signal.Notify(quit, os.Interrupt, syscall.SIGTERM)

	go func() {
		<-quit
		logger.Log().Info("Server is shutting down...")
		atomic.StoreInt32(&healthy, 0)

		ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
		defer cancel()

		server.SetKeepAlivesEnabled(false)
		if err := server.Shutdown(ctx); err != nil {
			logger.Log().Fatalf("Could not gracefully shutdown the server: %v\n", err)
		}
		close(done)
	}()

	logger.Log().Infof("Server is ready to handle requests at %+v", listenAddr)
	atomic.StoreInt32(&healthy, 1)

	listener, err := net.Listen(listenNetwork, listenAddr)
	if err != nil && err != http.ErrServerClosed {
		logger.Log().Fatalf("Could not listen on %s (network %s): %v\n", listenAddr, listenNetwork, err)
	}
	if err := server.Serve(listener); err != nil {
		logger.Log().Fatalf("Could not serve: %v\n", err)
	}
	<-done
	logger.Log().Info("Server stopped")
}
