package internal

import (
	"net/http"
	"net/http/httputil"
	"net/url"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
)

type loggedTransport struct {
	logger log.Logger
}

func (t *loggedTransport) RoundTrip(request *http.Request) (*http.Response, error) {
	response, err := http.DefaultTransport.RoundTrip(request)
	if err != nil {
		return nil, err
	}

	body, err := httputil.DumpResponse(response, true)
	if err != nil {
		return nil, err
	}

	responseText := truncateText(
		strings.ReplaceAll(string(body), "\n", "\\n"),
		1000,
	)
	t.logger.Debugf("Responding with %v", responseText)

	return response, err
}

func HandlerProxy(logger log.Logger) func(w http.ResponseWriter, r *http.Request) {
	return func(w http.ResponseWriter, r *http.Request) {
		targetURL, _ := url.Parse(r.Header.Get(ProxyTargetURLHeader))
		r.Header.Del(ProxyTargetURLHeader)

		l := log.With(
			logger,
			log.String("request_id", r.Header.Get("X-Request-Id")),
		)

		l.Debugf(
			"Proxying [Method: %v URL: %v] to %v for [Client IP: %v]",
			r.Method,
			r.URL,
			targetURL,
			r.Header.Get("X-Forwarded-For"),
		)
		proxy := httputil.ReverseProxy{Director: func(r *http.Request) {
			r.URL.Scheme = targetURL.Scheme
			r.URL.Host = targetURL.Host
			r.URL.Path = targetURL.Path + r.URL.Path
			r.Host = targetURL.Host
		}}
		proxy.Transport = &loggedTransport{logger: l}
		proxy.ServeHTTP(w, r)
	}
}
