package internal

import (
	"fmt"
	"net/http"
	"net/url"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
)

func stripPort(s string) string {
	ix := strings.IndexRune(s, ':')
	if ix == -1 {
		return s
	}
	return s[:ix]
}

func ValidateTargetHost(proxyCfg *Config, logger log.Logger) func(next http.Handler) http.Handler {
	allowedHosts := map[string]interface{}{}
	for _, host := range proxyCfg.AllowedTargetHosts {
		allowedHosts[host] = true
	}

	return func(next http.Handler) http.Handler {
		fn := func(w http.ResponseWriter, r *http.Request) {
			l := log.With(
				logger,
				log.String("request_id", r.Header.Get("X-Request-Id")),
			)

			targetURL, _ := url.Parse(r.Header.Get(ProxyTargetURLHeader))

			if _, ok := allowedHosts[stripPort(targetURL.Host)]; ok {
				next.ServeHTTP(w, r)
				return
			}
			l.Warnf("Proxying to host %s is not allowed, rejecting request %+v", targetURL.Host, *r)
			w.WriteHeader(http.StatusForbidden)
			response := []byte(fmt.Sprintf("Proxying to host %s is not allowed\n", targetURL.Host))
			_, err := w.Write(response)
			if err != nil {
				l.Errorf("Failed writing response: %+v", err)
			}
		}
		return http.HandlerFunc(fn)
	}
}
