package internal

import (
	"fmt"
	"net/http"

	"a.yandex-team.ru/library/go/core/log"
)

func VerifyProxyAuthorization(tvmAgent *TVM, logger log.Logger) func(next http.Handler) http.Handler {
	writeError := func(w http.ResponseWriter, response string) {
		w.WriteHeader(http.StatusProxyAuthRequired)
		_, err := w.Write([]byte(response))
		if err != nil {
			logger.Errorf("Failed writing response: %+v", err)
		}
	}
	return func(next http.Handler) http.Handler {
		fn := func(w http.ResponseWriter, r *http.Request) {
			l := log.With(
				logger,
				log.String("request_id", r.Header.Get("X-Request-Id")),
			)

			authHeader := r.Header.Get(ProxyTVMAuthorizationHeader)
			r.Header.Del(ProxyTVMAuthorizationHeader)
			if authHeader == "" {
				l.Warnf("TVM ticket is missing, rejecting request %+v", *r)
				writeError(w, "TVM ticket is missing")
				return
			}
			ticket, err := tvmAgent.client.CheckServiceTicket(r.Context(), authHeader)
			if err != nil {
				l.Warnf("TVM ticket is invalid, rejecting request %+v", *r)
				l.Warnf("Ticket data %+v, error %+v", ticket, err)
				writeError(w, "TVM ticket is invalid")
				return
			}
			if _, ok := tvmAgent.allowedIDs[ticket.SrcID]; !ok {
				l.Warnf("TVM client %d is not allowed, rejecting request %+v", ticket.SrcID, *r)
				writeError(w, fmt.Sprintf("TVM client %d is not allowed", ticket.SrcID))
				return
			}
			l.Debugf("Verified TVM ticket from %d", ticket.SrcID)
			next.ServeHTTP(w, r)
		}
		return http.HandlerFunc(fn)
	}
}
