package bbadapter

import (
	"context"
	"fmt"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/backend/scim_api/internal/bbutils"
	"a.yandex-team.ru/passport/backend/scim_api/internal/core/interfaces"
	"a.yandex-team.ru/passport/backend/scim_api/internal/core/models"
	"a.yandex-team.ru/passport/backend/scim_api/internal/logutils"
	passportBlackbox "a.yandex-team.ru/passport/shared/golibs/blackbox"
)

type blackboxAdapter struct {
	bb     passportBlackbox.Client
	logger log.Logger
}

func (a *blackboxAdapter) logCtx(ctx context.Context) log.Logger {
	return logutils.AddCommonFromContext(ctx, a.logger)
}

// compile-time проверка, что интерфейс имплементирован корректно
var _ interfaces.DomainsAdapter = (*blackboxAdapter)(nil)
var _ interfaces.AuthAdapter = (*blackboxAdapter)(nil)

func NewBlackboxAdapter(cfg bbutils.BlackboxConfig, tvmClient tvm.Client, logger log.Logger) *blackboxAdapter {
	return &blackboxAdapter{
		bb:     bbutils.NewBlackboxClient(cfg, tvmClient),
		logger: logger,
	}
}

func (a *blackboxAdapter) DomainExists(ctx context.Context, domainID uint64) (bool, error) {
	a.logCtx(ctx).Debugf("querying blackbox hosted domains for domain_id %d ...", domainID)
	domains, err := a.bb.HostedDomains(ctx, domainID)
	if err != nil {
		return false, fmt.Errorf("error querying blackbox for domain_id %d: %w", domainID, err)
	}
	for _, d := range domains.Domains {
		if d.DomainID == domainID {
			if d.Enabled {
				a.logCtx(ctx).Debugf("domain with domain_id %d exists", domainID)
				return true, nil
			} else {
				return false, fmt.Errorf("domain with domain_id %d is disabled", domainID)
			}
		}
	}
	return false, fmt.Errorf("domain with domain_id %d doesn't exists", domainID)
}

func (a *blackboxAdapter) CheckCredentials(ctx context.Context, credentials models.Credentials) (models.ClientInfo, error) {
	maskedToken := bbutils.MaskOAuthToken(credentials.Credential)
	a.logCtx(ctx).Debugf("[AuthAdapter] CheckCredentials(%s)", maskedToken)
	oauthRequest := blackbox.OAuthRequest{OAuthToken: credentials.Credential, UserIP: credentials.UserIP, Scopes: []string{"passport:scim-api.all"}}
	response, err := a.bb.OAuth(ctx, oauthRequest)
	if err != nil {
		return models.ClientInfo{}, fmt.Errorf("error checking oauth token %s in blackbox: %w", maskedToken, err)
	}
	if response.User.UID.ID != 0 {
		return models.ClientInfo{}, fmt.Errorf("token %s is not 2-legged but belongs to %d", maskedToken, response.User.UID.ID)
	}
	return models.ClientInfo{ClientID: response.ClientID}, nil
}
