package crud

import (
	"context"
	"fmt"
	"strconv"
	"strings"

	"github.com/elimity-com/scim"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/passport/backend/library/passportapi"
	"a.yandex-team.ru/passport/backend/scim_api/internal/core/interfaces"
	"a.yandex-team.ru/passport/backend/scim_api/internal/core/models"
	"a.yandex-team.ru/passport/backend/scim_api/internal/glue"
	"a.yandex-team.ru/passport/backend/scim_api/internal/logutils"
	passportBlackbox "a.yandex-team.ru/passport/shared/golibs/blackbox"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
)

type adapter struct {
	bb       passportBlackbox.Client
	passport passportapi.APIClient
	logger   log.Logger
}

// compile-time проверка, что интерфейс имплементирован корректно
var _ interfaces.SCIMAdapter = (*adapter)(nil)

func NewAdapter(bbClient passportBlackbox.Client, passportClient passportapi.APIClient, logger log.Logger) *adapter {
	return &adapter{
		bbClient,
		passportClient,
		logger,
	}
}

func (a *adapter) logCtx(ctx context.Context) log.Logger {
	return logutils.AddCommonFromContext(ctx, a.logger)
}

func getPassportHeaders(ctx context.Context) (passportapi.Headers, error) {
	headers := passportapi.Headers{}
	headers.YaClientXRequestID = middlewares.ContextReqID(ctx)
	headers.YaClientUserAgent = "passport-scim-api"
	clientIP, err := logutils.GetClientIP(ctx)
	if err != nil {
		return headers, fmt.Errorf("failed to get client-ip from ctx: %w", err)
	}
	headers.YaClientIP = clientIP
	return headers, nil
}

func (a *adapter) getDomainInfo(ctx context.Context, domainID uint64) (passportBlackbox.HostedDomain, error) {
	a.logCtx(ctx).Debugf("querying blackbox hosted domains for domain_id %d ...", domainID)
	domains, err := a.bb.HostedDomains(ctx, domainID)
	if err != nil {
		return passportBlackbox.HostedDomain{}, fmt.Errorf("error querying blackbox for domain_id %d: %w", domainID, err)
	}
	for _, d := range domains.Domains {
		if d.DomainID == domainID {
			return d, nil
		}
	}
	return passportBlackbox.HostedDomain{}, fmt.Errorf("domain with domain_id %d doesn't exists", domainID)
}

func (a *adapter) userinfoByUIDs(ctx context.Context, uids []uint64, domain string) ([]models.User, error) {
	var users []models.User
	clientIP, err := logutils.GetClientIP(ctx)
	if err != nil {
		return users, fmt.Errorf("error getting user %+v: %w", uids, err)
	}
	rsp, err := a.bb.UserInfo(ctx, blackbox.UserInfoRequest{
		UserIP:  clientIP,
		UIDs:    uids,
		RegName: true,
		Attributes: []blackbox.UserAttribute{
			blackbox.UserAttributePersonFirstname,
			blackbox.UserAttributePersonLastname,
			blackbox.UserAttributeAccountIsAvailable,
		},
		Aliases: []blackbox.UserAlias{
			blackbox.UserAliasPdd, // отсюда можно вытащить domain_id до слеша
			"24",                  // федеративный аккаунт
		},
		Emails: blackbox.EmailsGetAll,
	})
	if err != nil {
		return users, fmt.Errorf("error querying blackbox for uid %+v: %w", uids, err)
	}
	// TODO проверять ena 0/1

	// Выберем только федералов
	var federalUsersResp []blackbox.User
	for _, u := range rsp.Users {
		if u.Aliases["24"] != "" {
			federalUsersResp = append(federalUsersResp, u)
		} else {
			a.logCtx(ctx).Warnf("User %d is not federal. Skipping", u.UID.ID)
		}
	}

	users = make([]models.User, len(federalUsersResp))
	for i, u := range federalUsersResp {
		if err := glue.BlackboxUserToUser(u, &users[i]); err != nil {
			return users, fmt.Errorf("error getting user %+v: parsing struct from blackbox: %w", uids, err)
		}
		if domain != "" && !strings.Contains(users[i].UserName, "@") {
			users[i].UserName = users[i].UserName + "@" + domain
		}
	}
	return users, nil
}

func (a *adapter) userinfoByUserID(ctx context.Context, userID string, domain string) (models.User, error) {
	var user models.User
	uid, err := strconv.ParseUint(userID, 0, 64)
	if err != nil {
		return user, fmt.Errorf("error parsing uid from userID %+v: %w", userID, err)
	}
	users, err := a.userinfoByUIDs(ctx, []uint64{uid}, domain)
	if err != nil {
		return user, err
	}
	if len(users) == 0 {
		return user, fmt.Errorf("error getting user %s: %w", userID, interfaces.ErrUserNotFound)
	}
	if len(users) > 1 {
		return user, fmt.Errorf("error getting user %s: too many results: %d", userID, len(users))
	}

	return users[0], nil
}

func (a *adapter) RegisterUser(ctx context.Context, user models.User) (models.User, error) {
	var retval models.User

	req := passportapi.RegisterFederalRequest{
		FirstName:   user.FirstName,
		LastName:    user.LastName,
		Active:      user.IsActive,
		DomainID:    user.DomainID,
		DisplayName: user.DisplayName,
		NameID:      user.UserName,
	}
	if len(user.Emails) > 0 {
		// TODO надо выбирать адрес более интеллектуально
		req.EMail = user.Emails[0].Address
	}

	a.logCtx(ctx).Debugf("Sending params to passport-api: %+v", req)

	h, err := getPassportHeaders(ctx)
	if err != nil {
		return retval, fmt.Errorf("RegisterUser error: %w", err)
	}
	a.logCtx(ctx).Debugf("Sending headers to passport-api: %+v", h)
	rsp, err := a.passport.RegisterFederal(ctx, h, req)
	if err != nil {
		return retval, fmt.Errorf("RegisterUser passport-api error: %w", err)
	}
	if len(rsp.Errors) > 0 {
		for _, passportError := range rsp.Errors {
			if passportError == "email.exist" {
				return retval, fmt.Errorf("RegisterUser passport-api error: %+v: %w", rsp.Errors, interfaces.ErrConflict)
			}
		}
		return retval, fmt.Errorf("RegisterUser passport-api error: %+v: %s", rsp.Errors, rsp.ErrorMessage)
	}

	user.PassportUID = rsp.UID

	a.logCtx(ctx).Infof("RegisterUser %d: ok", user.PassportUID)

	return user, nil
}

func (a *adapter) PatchUser(ctx context.Context, userID string, patches []scim.PatchOperation) (models.User, error) {
	if len(patches) == 0 {
		a.logCtx(ctx).Infof("PatchUser: empty patch")
		return models.User{}, interfaces.ErrNoChanges
	}
	user, err := a.GetUser(ctx, userID)
	if err != nil {
		return user, err
	}
	madeChanges := applyPatches(&user, patches)
	if !madeChanges {
		a.logCtx(ctx).Infof("PatchUser %s: no changes", userID)
		return models.User{}, interfaces.ErrNoChanges
	}
	h, err := getPassportHeaders(ctx)
	if err != nil {
		return models.User{}, fmt.Errorf("PatchUser error %s: %w", userID, err)
	}
	a.logCtx(ctx).Infof("PatchUser: %s: %+v", userID, user)
	req := passportapi.EditFederalRequest{
		UID:         user.PassportUID,
		FirstName:   user.FirstName,
		LastName:    user.LastName,
		Active:      user.IsActive,
		DisplayName: user.DisplayName,
	}
	if len(user.Emails) > 0 {
		// TODO надо выбирать адреса более интеллектуально
		req.EMails = []string{user.Emails[0].Address}
	}

	rsp, err := a.passport.EditFederal(ctx, h, req)
	if err != nil {
		return user, fmt.Errorf("PatchUser passport-api error %s: %w", userID, err)
	}
	if len(rsp.Errors) > 0 {
		for _, passportError := range rsp.Errors {
			if passportError == "email.exist" {
				return user, fmt.Errorf("PatchUser passport-api error %s: %+v: %w", userID, rsp.Errors, interfaces.ErrConflict)
			}
		}
		return user, fmt.Errorf("PatchUser passport-api error %s: %+v: %s", userID, rsp.Errors, rsp.ErrorMessage)
	}
	return user, nil
}

func (a *adapter) GetUser(ctx context.Context, userID string) (models.User, error) {
	var user models.User

	domainID, err := logutils.GetDomainID(ctx)
	if err != nil {
		return models.User{}, fmt.Errorf("GetUser %s error: %w", userID, err)
	}
	if domainID == 0 {
		panic("oxoxo")
	}
	hostedDomain, err := a.getDomainInfo(ctx, domainID)
	if err != nil {
		return models.User{}, fmt.Errorf("GetUser %s error: error during hosted_domains %d: %w", userID, domainID, err)
	}
	user, err = a.userinfoByUserID(ctx, userID, hostedDomain.Domain)
	if err != nil {
		return user, fmt.Errorf("GetUser %s error: %w", userID, err)
	}
	a.logCtx(ctx).Infof("GetUser %s: ok", userID)
	return user, nil
}

func (a *adapter) ReplaceUser(ctx context.Context, userID string, user models.User) (models.User, error) {
	panic("not implemented")
}

func (a *adapter) DeleteUser(ctx context.Context, userID string) error {
	// должен дергать ручку на удаление аккаунта в паспорте
	user, err := a.userinfoByUserID(ctx, userID, "")
	if err != nil {
		return fmt.Errorf("DeleteUser %s error: %w", userID, err)
	}
	h, err := getPassportHeaders(ctx)
	if err != nil {
		return fmt.Errorf("DeleteUser %s error: %w", userID, err)
	}
	a.logCtx(ctx).Debugf("Sending headers to passport-api: %+v", h)
	rsp, err := a.passport.DeleteAccount(ctx, h, user.PassportUID)
	if err != nil {
		return fmt.Errorf("DeleteUser %s passport-api error: %w", userID, err)
	}
	if len(rsp.Errors) > 0 {
		return fmt.Errorf("DeleteUser %s passport-api errors: %+v", userID, rsp.Errors)
	}
	a.logCtx(ctx).Infof("DeleteUser %s: ok", userID)
	return nil
}

func (a *adapter) ListUsers(ctx context.Context, startIndex uint64, limit uint64) (uint64, []models.User, error) {
	domainID, err := logutils.GetDomainID(ctx)
	if err != nil {
		return 0, []models.User{}, fmt.Errorf("error listing users: %w", err)
	}
	if domainID == 0 {
		panic("oxoxo")
	}
	hostedDomain, err := a.getDomainInfo(ctx, domainID)
	if err != nil {
		return 0, []models.User{}, fmt.Errorf("error getting domain info %d: error during hosted_domains: %w", domainID, err)
	}
	key := passportBlackbox.FindPddAccountsSort{Key: passportBlackbox.FindPddAccountsKeyUID, Order: passportBlackbox.FindPddAccountsOrderAsc}
	rsp, err := a.bb.FindPddAccounts(ctx, domainID, startIndex, limit, key)
	if err != nil {
		return 0, []models.User{}, fmt.Errorf("error listing users %d: error during find_pdd_accounts: %w", domainID, err)
	}
	a.logCtx(ctx).Debugf("listing users %d: got %d uids", domainID, len(rsp.UIDs))
	if len(rsp.UIDs) == 0 {
		return rsp.TotalCount, []models.User{}, nil
	}
	users, err := a.userinfoByUIDs(ctx, rsp.UIDs, hostedDomain.Domain)
	if err != nil {
		return 0, []models.User{}, fmt.Errorf("error listing users %d: error during userinfo: %w", domainID, err)
	}
	return rsp.TotalCount, users, err
}
