package inmemory

import (
	"context"
	"fmt"
	"math/rand"
	"time"

	"github.com/elimity-com/scim"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/passport/backend/scim_api/internal/core/interfaces"
	"a.yandex-team.ru/passport/backend/scim_api/internal/core/models"
	"a.yandex-team.ru/passport/backend/scim_api/internal/logutils"
)

type adapter struct {
	data   map[string]models.User
	logger log.Logger
}

// compile-time проверка, что интерфейс имплементирован корректно
var _ interfaces.SCIMAdapter = (*adapter)(nil)

func NewAdapter(logger log.Logger) *adapter {
	return &adapter{
		map[string]models.User{},
		logger,
	}
}

func (a *adapter) logCtx(ctx context.Context) log.Logger {
	return logutils.AddCommonFromContext(ctx, a.logger)
}

func (a *adapter) RegisterUser(ctx context.Context, user models.User) (models.User, error) {
	// должен дергать ручку паспорта на создание пользователя
	rand.Seed(time.Now().UnixNano())
	id := rand.Uint64()

	user.PassportUID = id
	user.IsActive = true

	a.data[fmt.Sprintf("%d", id)] = user

	a.logCtx(ctx).Infof("RegisterUser: ok")

	return user, nil
}

func (a *adapter) PatchUser(ctx context.Context, userID string, patches []scim.PatchOperation) (models.User, error) {
	// должен дергать ручку паспорта на изменение пользователя
	var madeChanges bool
	if len(patches) == 0 {
		a.logCtx(ctx).Infof("PatchUser: empty patch")
		return models.User{}, interfaces.ErrNoChanges
	}
	user, err := a.GetUser(ctx, userID)
	for _, patch := range patches {
		madeChanges = applySinglePatch(&user, patch)
	}
	if madeChanges {
		a.data[userID] = user
		a.logCtx(ctx).Infof("PatchUser: ok")
	} else {
		a.logCtx(ctx).Infof("PatchUser: no changes")
		return models.User{}, interfaces.ErrNoChanges
	}
	return user, err
}

func (a *adapter) GetUser(ctx context.Context, userID string) (models.User, error) {
	// должен дергать ЧЯ method=userinfo
	user, ok := a.data[userID]
	if !ok {
		a.logCtx(ctx).Infof("GetUser: not found")
		return models.User{}, interfaces.ErrUserNotFound
	}
	a.logCtx(ctx).Infof("GetUser: ok")
	return user, nil
}

func (a *adapter) ReplaceUser(ctx context.Context, userID string, user models.User) (models.User, error) {
	// должен дергать ручку паспорта на изменение пользователя (так же как в Patch)
	_, ok := a.data[userID]
	if !ok {
		a.logCtx(ctx).Infof("ReplaceUser: user not found")
		return models.User{}, interfaces.ErrUserNotFound
	}

	a.data[userID] = user
	a.logCtx(ctx).Infof("ReplaceUser: ok")
	return user, nil
}

func (a *adapter) DeleteUser(ctx context.Context, userID string) error {
	// должен дергать ручку на удаление аккаунта в паспорте
	_, ok := a.data[userID]
	if !ok {
		a.logCtx(ctx).Infof("DeleteUser: user not found")
		return interfaces.ErrUserNotFound
	}
	delete(a.data, userID)
	a.logCtx(ctx).Infof("DeleteUser: ok")
	return nil
}

func (a *adapter) ListUsers(ctx context.Context, startIndex uint64, limit uint64) (uint64, []models.User, error) {
	// должен дергать ЧЯ method=find_pdd_account
	if limit == 0 {
		return 0, []models.User{}, nil
	}

	users := make([]models.User, 0)
	i := uint64(1)

	for _, v := range a.data {
		if i > (startIndex + limit - 1) {
			break
		}

		if i >= startIndex {
			users = append(users, v)
		}
		i++
	}

	return uint64(len(a.data)), users, nil
}
