package authcontroller

import (
	"context"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/passport/backend/scim_api/internal/core/interfaces"
	"a.yandex-team.ru/passport/backend/scim_api/internal/core/models"
	"a.yandex-team.ru/passport/backend/scim_api/internal/logutils"
)

type controller struct {
	authAdapter    interfaces.AuthAdapter
	domainsAdapter interfaces.DomainsAdapter
	fedCfgAdapter  interfaces.FederationConfigAdapter
	logger         log.Logger
}

// compile-time проверка, что интерфейс имплементирован корректно
var _ interfaces.AuthController = (*controller)(nil)

func NewController(domainsAdapter interfaces.DomainsAdapter, authAdapter interfaces.AuthAdapter, fedCfgAdapter interfaces.FederationConfigAdapter, logger log.Logger) *controller {
	return &controller{
		authAdapter:    authAdapter,
		domainsAdapter: domainsAdapter,
		fedCfgAdapter:  fedCfgAdapter,
		logger:         logger,
	}
}

func (ctrl *controller) logCtx(ctx context.Context) log.Logger {
	return logutils.AddCommonFromContext(ctx, ctrl.logger)
}

func (ctrl *controller) CanOperateOnResources(ctx context.Context, credentials models.Credentials, domainID uint64) error {
	// проверяем, что токен валиден
	clientInfo, err := ctrl.authAdapter.CheckCredentials(ctx, credentials)
	if err != nil {
		return err
	}
	ctrl.logCtx(ctx).Debugf("Successfully checked credentials")
	// проверяем живость домена
	ok, err := ctrl.domainsAdapter.DomainExists(ctx, domainID)
	if err != nil {
		return err
	}
	if !ok {
		return interfaces.ErrNoDomain
	}
	fedCfg, err := ctrl.fedCfgAdapter.GetConfigByDomainID(ctx, domainID)
	if err != nil {
		ctrl.logCtx(ctx).Warnf("%s", err.Error())
		return interfaces.ErrNoConfig
	}
	if !fedCfg.Enabled {
		return interfaces.ErrConfigDisabled
	}
	if fedCfg.OAuthClient != clientInfo.ClientID {
		ctrl.logCtx(ctx).Warnf("expected client_id=%s but got client_id=%s", fedCfg.OAuthClient, clientInfo.ClientID)
		return interfaces.ErrClientMismatch
	}
	return nil
}
