package glue

import (
	"fmt"
	"strconv"
	"strings"
	"time"

	"github.com/elimity-com/scim"
	"github.com/elimity-com/scim/optional"

	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/passport/backend/scim_api/internal/core/models"
)

func parse24Alias(alias string) (uint64, string, error) {
	if len(alias) == 0 {
		return 0, "", fmt.Errorf("error parsing alias: empty string")
	}
	idx := strings.Index(alias, "/")
	if idx <= 0 {
		return 0, "", fmt.Errorf("error parsing alias, couldn't find domain_id: %s", alias)
	}
	nameID := alias[idx+1:]
	if len(nameID) == 0 {
		return 0, "", fmt.Errorf("error parsing alias, couldn't find name_id: %s", alias)
	}
	strDomainID := alias[:idx]
	domainID, err := strconv.ParseUint(strDomainID, 0, 64)
	if err != nil {
		return 0, "", fmt.Errorf("error parsing alias, couldn't convert domain_id to int: %w", err)
	}
	return domainID, nameID, nil
}

func BlackboxUserToUser(blackboxUser blackbox.User, user *models.User) error {
	user.PassportUID = blackboxUser.UID.ID
	user.FirstName = blackboxUser.Attributes[blackbox.UserAttributePersonFirstname]
	user.LastName = blackboxUser.Attributes[blackbox.UserAttributePersonLastname]
	for _, addr := range blackboxUser.AddressList {
		user.Emails = append(user.Emails, models.Email{Address: addr.Address})
	}
	domainID, nameID, err := parse24Alias(blackboxUser.Aliases["24"])
	if err != nil {
		return fmt.Errorf("error parsing user %d: %w", user.PassportUID, err)
	}
	user.DisplayName = blackboxUser.DisplayName.Name
	user.DomainID = domainID
	user.UserName = nameID
	if val, ok := blackboxUser.Attributes[blackbox.UserAttributeAccountIsAvailable]; ok {
		user.IsActive = val == "1"
	} else {
		user.IsActive = false
	}
	return nil
}

func AttributesToUser(attributes scim.ResourceAttributes, user *models.User) error {
	v, ok := attributes["active"]
	if ok {
		user.IsActive = v.(bool)
	} else {
		user.IsActive = true
	}

	v, ok = attributes["userName"]
	if ok {
		user.UserName = v.(string)
	}
	v, ok = attributes["displayName"]
	if ok {
		user.DisplayName = v.(string)
	}
	v, ok = attributes["externalID"]
	if ok {
		user.SCIMAttributes.ExternalID = v.(string)
	}
	emails, ok := attributes["emails"]
	if ok {
		for _, emailStruct := range emails.([]interface{}) {
			emailAddress, okValue := emailStruct.(map[string]interface{})["value"]
			emailIsPrimary, okPrimary := emailStruct.(map[string]interface{})["Primary"]
			if okValue {
				user.Emails = append(user.Emails, models.Email{
					Address:   emailAddress.(string),
					IsPrimary: okPrimary && emailIsPrimary.(bool),
				})
			}
		}
	}
	phoneNumbers, ok := attributes["phoneNumbers"]
	if ok {
		for _, phoneStruct := range phoneNumbers.([]interface{}) {
			phoneNumber, ok := phoneStruct.(map[string]interface{})["value"]
			if ok {
				user.PhoneNumbers = append(user.PhoneNumbers, models.PhoneNumber{
					PhoneNumber: phoneNumber.(string),
				})
			}
		}
	}
	nameStruct, ok := attributes["name"]
	if ok {
		formatted, ok := nameStruct.(map[string]interface{})["formatted"]
		if ok && user.DisplayName == "" {
			user.DisplayName = formatted.(string)
		}
		givenName, ok := nameStruct.(map[string]interface{})["givenName"]
		if ok {
			user.FirstName = givenName.(string)
		}
		familyName, ok := nameStruct.(map[string]interface{})["familyName"]
		if ok {
			user.LastName = familyName.(string)
		}
	}

	return nil
}

func UserToResource(user models.User) (scim.Resource, error) {
	var res scim.Resource

	res.ID = strconv.FormatUint(uint64(user.PassportUID), 10)
	if user.SCIMAttributes.ExternalID != "" {
		res.ExternalID = optional.NewString(user.SCIMAttributes.ExternalID)
	}

	created, _ := time.ParseInLocation(time.RFC3339, fmt.Sprintf("%v", user.SCIMAttributes.CreatedTime), time.UTC)
	lastModified, _ := time.Parse(time.RFC3339, fmt.Sprintf("%v", user.SCIMAttributes.ModifiedTime))

	res.Attributes = scim.ResourceAttributes{}
	res.Attributes["userName"] = user.UserName
	res.Attributes["active"] = user.IsActive
	res.Attributes["displayName"] = user.DisplayName
	res.Attributes["id"] = fmt.Sprintf("%d", user.PassportUID)

	res.Attributes["name"] = map[string]string{
		"givenName":  user.FirstName,
		"familyName": user.LastName,
	}
	if len(user.Emails) > 0 {
		var emails []map[string]interface{}
		for _, emailStruct := range user.Emails {
			emailJSON := map[string]interface{}{
				"Primary": emailStruct.IsPrimary,
				"value":   emailStruct.Address,
			}

			emails = append(
				emails,
				emailJSON,
			)
		}

		res.Attributes["emails"] = emails
	}
	if len(user.PhoneNumbers) > 0 {
		var phoneNumbers []map[string]interface{}
		for _, phoneStruct := range user.PhoneNumbers {
			phoneJSON := map[string]interface{}{
				"Primary": false,
				"value":   phoneStruct.PhoneNumber,
			}

			phoneNumbers = append(
				phoneNumbers,
				phoneJSON,
			)
		}

		res.Attributes["phoneNumbers"] = phoneNumbers
	}

	res.Meta.Version = fmt.Sprintf("v%s", user.SCIMAttributes.ExternalID)
	res.Meta.Created = &created
	res.Meta.LastModified = &lastModified

	return res, nil
}
