package scimapi

import (
	"fmt"
	"net/http"
	"net/url"
	"time"

	"github.com/elimity-com/scim"
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/backend/library/passportapi"
	"a.yandex-team.ru/passport/backend/scim_api/internal/adapters/bbadapter"
	"a.yandex-team.ru/passport/backend/scim_api/internal/adapters/crud"
	"a.yandex-team.ru/passport/backend/scim_api/internal/adapters/fedcfg"
	"a.yandex-team.ru/passport/backend/scim_api/internal/bbutils"
	"a.yandex-team.ru/passport/backend/scim_api/internal/core/controllers/authcontroller"
	"a.yandex-team.ru/passport/backend/scim_api/internal/core/controllers/users"
	"a.yandex-team.ru/passport/backend/scim_api/internal/fedcfgclient"
	"a.yandex-team.ru/passport/backend/scim_api/internal/logutils"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/httpdtvm"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
)

type ScimAPI struct {
	tvm        tvm.Client
	unistat    stats
	scimServer scim.Server
	AppLoggers
}

type AppLoggers struct {
	accessLog   log.Logger
	blackboxLog log.Logger
	scimLog     log.Logger
	fedCfgLog   log.Logger
	handlerLog  log.Logger
	authLog     log.Logger
}

type Config struct {
	Tvm         httpdtvm.TvmConfig     `json:"tvm"`
	AccessLog   logger.Config          `json:"access_log"`
	BlackboxLog logger.Config          `json:"blackbox_log"`
	ScimLog     logger.Config          `json:"scim_log"`
	FedCfgLog   logger.Config          `json:"fedcfg_log"`
	HandlerLog  logger.Config          `json:"handler_log"`
	AuthLog     logger.Config          `json:"auth_log"`
	Blackbox    bbutils.BlackboxConfig `json:"blackbox"`
	PassportAPI PassportAPIConfig      `json:"passport_api"`
	FedCfgAPI   FedCfgAPIConfig        `json:"fedcfg_api"`
}

type PassportAPIConfig struct {
	Env             string `json:"env"`
	PassportTimeout string `json:"timeout"`
	Consumer        string `json:"consumer"`
}

type FedCfgAPIConfig struct {
	URL     string       `json:"url"`
	Timeout string       `json:"timeout"`
	TVMID   tvm.ClientID `json:"tvm_id"`
}

type stats struct {
	requestsByPath *unistat.SignalSet
}

type Factory struct{}

const (
	emptyString       = "-"
	headerETag        = "ETag"
	headerIfNoneMatch = "If-None-Match"
)

func (f *Factory) CreateLoggers(cfg Config) (AppLoggers, error) {
	var loggers AppLoggers

	accessLog, err := logger.CreateLog(cfg.AccessLog)
	if err != nil {
		return loggers, err
	}

	blackboxLog, err := logger.CreateLog(cfg.BlackboxLog)
	if err != nil {
		return loggers, err
	}

	scimLog, err := logger.CreateLog(cfg.ScimLog)
	if err != nil {
		return loggers, err
	}

	fedCfgLog, err := logger.CreateLog(cfg.FedCfgLog)
	if err != nil {
		return loggers, err
	}

	handlerLog, err := logger.CreateLog(cfg.HandlerLog)
	if err != nil {
		return loggers, err
	}

	authLog, err := logger.CreateLog(cfg.AuthLog)
	if err != nil {
		return loggers, err
	}

	loggers.accessLog = log.With(accessLog, log.String("loggerName", "access_log"))
	loggers.blackboxLog = log.With(blackboxLog, log.String("loggerName", "blackbox_log"))
	loggers.scimLog = log.With(scimLog, log.String("loggerName", "scim_log"))
	loggers.fedCfgLog = log.With(fedCfgLog, log.String("loggerName", "fedcfg_log"))
	loggers.handlerLog = log.With(handlerLog, log.String("loggerName", "handler_log"))
	loggers.authLog = log.With(authLog, log.String("loggerName", "auth_log"))

	return loggers, nil
}

func newPassportAPIClient(cfg PassportAPIConfig, tvmClient tvm.Client) passportapi.APIClient {
	timeout, err := time.ParseDuration(cfg.PassportTimeout)
	if err != nil {
		panic(err)
	}
	var env passportapi.Environment
	switch cfg.Env {
	case "testing":
		env = passportapi.EnvTesting
	case "production":
		env = passportapi.EnvProduction
	default:
		panic(fmt.Sprintf("unexpected passport env: %s", cfg.Env))
	}
	client, err := passportapi.NewPassportAPIClient(cfg.Consumer, tvmClient, env, timeout)
	if err != nil {
		panic(err)
	}
	return client
}

func newFedCfgAPIClient(cfg FedCfgAPIConfig, tvmClient tvm.Client) fedcfgclient.APIClient {
	timeout, err := time.ParseDuration(cfg.Timeout)
	if err != nil {
		panic(err)
	}
	client, err := fedcfgclient.NewFedCfgAPIClient(tvmClient, cfg.TVMID, cfg.URL, timeout)
	if err != nil {
		panic(err)
	}
	return client
}

func (f *Factory) NewService(config httpdaemon.ServiceConfig) (httpdaemon.Service, error) {
	var cfg Config
	if err := httpdaemon.ParseServiceConfig(config, &cfg); err != nil {
		return nil, err
	}

	tvmClient, err := httpdtvm.InitTvm(cfg.Tvm)
	if err != nil {
		return nil, err
	}

	loggers, err := f.CreateLoggers(cfg)
	if err != nil {
		return nil, err
	}

	// инициализируем контроллеры и адаптеры
	bbAdapter := bbadapter.NewBlackboxAdapter(cfg.Blackbox, tvmClient, loggers.blackboxLog)
	passportAPIClient := newPassportAPIClient(cfg.PassportAPI, tvmClient)
	crudAdapter := crud.NewAdapter(bbutils.NewBlackboxClient(cfg.Blackbox, tvmClient), passportAPIClient, loggers.scimLog)
	usersController := users.NewController(crudAdapter)
	fedCfgAPIClient := newFedCfgAPIClient(cfg.FedCfgAPI, tvmClient)
	fedCfgAdapter := fedcfg.New(fedCfgAPIClient, loggers.fedCfgLog)
	authController := authcontroller.NewController(
		bbAdapter,
		bbAdapter,
		fedCfgAdapter,
		loggers.authLog,
	)

	res := &ScimAPI{
		tvm: tvmClient,
		unistat: stats{
			requestsByPath: unistat.DefaultChunk.CreateSignalSet("in.requests."),
		},
		scimServer: GetScimServer(usersController, authController, loggers.handlerLog),
		AppLoggers: loggers,
	}

	return res, nil
}

func (t *ScimAPI) AddHandlers(e *echo.Echo) {
	e.Pre(
		t.middlewarePopulateRequest(),
		t.middlewareAccessLog(),
	)

	e.GET(
		"/ping",
		t.HandlePing(),
	)

	e.GET("/v2/*",
		echo.WrapHandler(
			http.HandlerFunc(t.scimServer.ServeHTTP)),
	)
	e.POST("/v2/*",
		echo.WrapHandler(
			http.HandlerFunc(t.scimServer.ServeHTTP)),
	)
	e.PATCH("/v2/*",
		echo.WrapHandler(
			http.HandlerFunc(t.scimServer.ServeHTTP)),
	)
	e.PUT("/v2/*",
		echo.WrapHandler(
			http.HandlerFunc(t.scimServer.ServeHTTP)),
	)
	e.DELETE("/v2/*",
		echo.WrapHandler(
			http.HandlerFunc(t.scimServer.ServeHTTP)),
	)
}

func (t *ScimAPI) HandlePing() echo.HandlerFunc {
	return func(c echo.Context) error {
		l := logutils.AddCommonFromContext(c.Request().Context(), t.handlerLog)
		l.Debugf("Ping: service is up")
		return c.String(http.StatusOK, "")
	}
}

func (t *ScimAPI) middlewarePopulateRequest() echo.MiddlewareFunc {
	// Мидлварь, складывающая в контекст общие для запроса штуки,
	// будь то client_ip или возможно еще чего-то
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) error {
			var domainID uint64
			if c.Request().Host != "localhost" && c.Request().Host != "localhost:80" {
				var err error
				domainID, err = getDomainIDFromHost(c.Request())
				if err != nil {
					t.handlerLog.Warnf("failed to get domain_id from host header: %s", err.Error())
					return c.String(http.StatusNotFound, "missing domain id")
				}
			}

			ctxRequest := logutils.CtxRequest{
				ClientIP: c.Request().Header.Get("X-Real-IP"),
				DomainID: domainID,
			}
			queryParams, err := url.ParseQuery(c.Request().URL.RawQuery)
			if err == nil {
				ctxRequest.ExternalRequestID = queryParams.Get("request_id")
				if ctxRequest.ExternalRequestID == "" {
					// параметр от наших функциональных тестов
					ctxRequest.ExternalRequestID = queryParams.Get("test_req_id")
				}
			}

			c.SetRequest(c.Request().WithContext(
				logutils.AddRequestToContext(
					c.Request().Context(),
					&ctxRequest,
				),
			))
			return next(c)
		}
	}
}

func (t *ScimAPI) middlewareAccessLog() echo.MiddlewareFunc {
	nullable := func(val string) string {
		if len(val) == 0 {
			return emptyString
		}
		return val
	}

	consumer := func(t *tvm.CheckedServiceTicket) string {
		if t == nil {
			return emptyString
		}
		return fmt.Sprintf("%d", t.SrcID)
	}

	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			c.Response().After(func() {
				startTime := middlewares.ContextStartInstant(c.Request().Context())
				st := tvm.ContextServiceTicket(c.Request().Context())

				l := logutils.AddCommonFromContext(c.Request().Context(), t.accessLog)
				l.Infof(
					"%s\t%s\t%s\t%s\t%s\t%.1fms\t%d\t%d\t%s\t%s\t%s",
					middlewares.ContextReqID(c.Request().Context()),
					c.RealIP(),
					consumer(st),
					c.Request().Method,
					c.Request().URL.Path,
					float64(time.Since(startTime).Microseconds())/1000,
					c.Response().Status,
					c.Response().Size,
					nullable(c.Request().Header.Get(headerIfNoneMatch)),
					nullable(c.Request().URL.RawQuery),
					nullable(c.Response().Header().Get(headerETag)),
				)
			})

			return next(c)
		}
	}
}
