package scimapi

import (
	"context"
	"errors"
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"time"

	"github.com/elimity-com/scim"
	scimErrors "github.com/elimity-com/scim/errors"
	"github.com/elimity-com/scim/optional"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/passport/backend/scim_api/internal/core/interfaces"
	"a.yandex-team.ru/passport/backend/scim_api/internal/core/models"
	"a.yandex-team.ru/passport/backend/scim_api/internal/glue"
	"a.yandex-team.ru/passport/backend/scim_api/internal/logutils"
)

type userResourceHandler struct {
	usersController interfaces.UsersController
	authController  interfaces.AuthController
	logger          log.Logger
}

var errTimeout interface{ Timeout() bool }

func isTimeout(err error) bool {
	return errors.As(err, &errTimeout)
}

func getDomainIDFromHost(r *http.Request) (uint64, error) {
	bits := strings.Split(r.Host, ".")
	if len(bits) == 1 {
		return 0, fmt.Errorf("couldn't split host header by . char: %s", r.Host)
	}
	domainID, err := strconv.ParseUint(bits[0], 0, 64)
	if err != nil {
		return 0, fmt.Errorf("failed to parse domain_id from host header: %s, %w", r.Host, err)
	}
	return domainID, nil
}

func getDomainIDFromHeaders(r *http.Request) (uint64, error) {
	domainIDHeader := r.Header.Get("X-Ya-Domain-ID")
	if len(domainIDHeader) == 0 {
		return 0, errors.New("missing x-ya-domain-id header")
	}
	domainID, err := strconv.ParseUint(domainIDHeader, 0, 64)
	if err != nil {
		return 0, fmt.Errorf("failed to parse domain_id from x-ya-domain-id header: %s, %w", r.Host, err)
	}
	return domainID, nil
}

func getDomainID(r *http.Request) uint64 {
	domainID, err := getDomainIDFromHost(r)
	if err != nil {
		domainID, err = getDomainIDFromHeaders(r)
		if err != nil {
			panic(err)
		}
	}
	return domainID
}

func getToken(r *http.Request) (string, error) {
	authHeader := r.Header.Get("Authorization")
	if len(authHeader) == 0 {
		return "", errors.New("missing authorization header")
	}
	if !strings.HasPrefix(authHeader, "Bearer ") {
		return "", errors.New("expected bearer auth method")
	}
	token := strings.TrimSpace(authHeader[len("Bearer "):])
	if len(token) == 0 {
		return "", errors.New("empty token")
	}
	return token, nil
}

func getClientIP(r *http.Request) (string, error) {
	clientIP := r.Header.Get("X-Real-IP")
	if clientIP == "" {
		return clientIP, errors.New("missing x-real-ip header or empty value")
	}
	return clientIP, nil
}

func (h userResourceHandler) logCtx(r *http.Request) log.Logger {
	return logutils.AddCommonFromContext(r.Context(), h.logger)
}

func (h userResourceHandler) AssertAllowed(r *http.Request) error {
	domainID, err := logutils.GetDomainID(r.Context())
	if err != nil {
		return scimErrors.ScimErrorBadRequest(err.Error())
	}
	h.logCtx(r).Debugf("using domain_id %d", domainID)
	token, err := getToken(r)
	if err != nil {
		return scimErrors.ScimError{
			Status: http.StatusUnauthorized,
			Detail: err.Error(),
		}
	}
	clientIP, err := getClientIP(r)
	if err != nil {
		return scimErrors.ScimErrorBadRequest(err.Error())
	}
	creds := models.Credentials{
		Credential: token,
		UserIP:     clientIP,
	}
	if err := h.authController.CanOperateOnResources(r.Context(), creds, domainID); err != nil {
		h.logCtx(r).Warnf("forbidden operation on domain %d: %s", domainID, err.Error())
		if isTimeout(err) || errors.Is(err, context.DeadlineExceeded) || errors.Is(err, context.Canceled) {
			return scimErrors.ScimError{
				Status: http.StatusInternalServerError,
				Detail: "auth backend is unavailable",
			}
		} else {
			return scimErrors.ScimError{
				Status: http.StatusForbidden,
				Detail: "forbidden",
			}
		}
	}

	return err
}

func (h userResourceHandler) Create(r *http.Request, attributes scim.ResourceAttributes) (scim.Resource, error) {
	if err := h.AssertAllowed(r); err != nil {
		return scim.Resource{}, err
	}

	var user models.User

	now := time.Now()

	if err := glue.AttributesToUser(attributes, &user); err != nil {
		panic(err)
	}
	user.SCIMAttributes.CreatedTime = now
	user.SCIMAttributes.ModifiedTime = now
	user.DomainID = getDomainID(r)

	user, err := h.usersController.RegisterUser(r.Context(), user)
	if err != nil {
		h.logCtx(r).Warnf("create user error: %s", err.Error())
		if isTimeout(err) || errors.Is(err, context.DeadlineExceeded) || errors.Is(err, context.Canceled) {
			return scim.Resource{}, scimErrors.ScimError{
				Status: http.StatusInternalServerError,
				Detail: "passport backend is unavailable",
			}
		} else if errors.Is(err, interfaces.ErrConflict) {
			return scim.Resource{}, scimErrors.ScimError{
				Status:   http.StatusConflict,
				Detail:   "entity already exists",
				ScimType: scimErrors.ScimTypeUniqueness,
			}
		}
		return scim.Resource{}, scimErrors.ScimErrorInternal
	}

	return scim.Resource{
		ID:         fmt.Sprintf("%d", user.PassportUID),
		ExternalID: optional.String{},
		Attributes: attributes,
		Meta: scim.Meta{
			Created:      &user.SCIMAttributes.CreatedTime,
			LastModified: &user.SCIMAttributes.ModifiedTime,
			Version:      fmt.Sprintf("v%s", user.SCIMAttributes.ExternalID),
		},
	}, nil
}

func (h userResourceHandler) Get(r *http.Request, id string) (scim.Resource, error) {
	if err := h.AssertAllowed(r); err != nil {
		return scim.Resource{}, err
	}

	var (
		user models.User
		res  scim.Resource
	)

	user, err := h.usersController.GetUser(r.Context(), id)
	if err != nil {
		if errors.Is(err, interfaces.ErrUserNotFound) {
			h.logCtx(r).Infof("user %s not found: %s", id, err.Error())
			return scim.Resource{}, scimErrors.ScimErrorResourceNotFound(id)
		} else {
			h.logCtx(r).Warnf("error getting user %s: %s", id, err.Error())
			return scim.Resource{}, scimErrors.ScimErrorInternal
		}
	}

	res, err = glue.UserToResource(user)
	return res, err
}

func (h userResourceHandler) Delete(r *http.Request, id string) error {
	if err := h.AssertAllowed(r); err != nil {
		return err
	}

	err := h.usersController.DeleteUser(r.Context(), id)
	if err != nil {
		h.logCtx(r).Warnf("error deleting user %s: %s", id, err.Error())
		if isTimeout(err) || errors.Is(err, context.DeadlineExceeded) || errors.Is(err, context.Canceled) {
			return scimErrors.ScimError{
				Status: http.StatusInternalServerError,
				Detail: "passport backend is unavailable",
			}
		} else if errors.Is(err, interfaces.ErrUserNotFound) {
			return scimErrors.ScimErrorResourceNotFound(id)
		} else {
			return scimErrors.ScimErrorInternal
		}
	}
	return nil
}

func (h userResourceHandler) GetAll(r *http.Request, params scim.ListRequestParams) (scim.Page, error) {
	if err := h.AssertAllowed(r); err != nil {
		return scim.Page{}, err
	}
	count := uint64(params.Count)
	if count == 0 {
		count = 100
	}
	offset := uint64(params.StartIndex) - 1
	total, users, err := h.usersController.ListUsers(r.Context(), offset, count)
	if err != nil {
		h.logCtx(r).Warnf("error listing  users: %s", err.Error())
		return scim.Page{}, scimErrors.ScimErrorInternal
	}

	resources := make([]scim.Resource, 0)
	for _, user := range users {
		resource, err := glue.UserToResource(user)
		if err != nil {
			h.logCtx(r).Warnf("error listing  users: rendering user to resource: %s", err.Error())
			return scim.Page{}, scimErrors.ScimErrorInternal
		}
		resources = append(resources, resource)
	}

	return scim.Page{
		TotalResults: int(total),
		Resources:    resources,
	}, nil
}

func (h userResourceHandler) Patch(r *http.Request, id string, operations []scim.PatchOperation) (scim.Resource, error) {
	if err := h.AssertAllowed(r); err != nil {
		return scim.Resource{}, err
	}

	user, err := h.usersController.PatchUser(r.Context(), id, operations)
	if err != nil {
		h.logCtx(r).Warnf("error patching user %s: %s", id, err.Error())
		if errors.Is(err, interfaces.ErrNoChanges) {
			return scim.Resource{}, nil
		} else if errors.Is(err, interfaces.ErrUserNotFound) {
			return scim.Resource{}, scimErrors.ScimErrorResourceNotFound(id)
		} else {
			return scim.Resource{}, scimErrors.ScimErrorInternal
		}
	}
	return glue.UserToResource(user)
}

func (h userResourceHandler) Replace(r *http.Request, id string, attributes scim.ResourceAttributes) (scim.Resource, error) {
	if err := h.AssertAllowed(r); err != nil {
		return scim.Resource{}, err
	}

	var user models.User
	err := glue.AttributesToUser(attributes, &user)
	if err != nil {
		return scim.Resource{}, scimErrors.ScimErrorInternal
	}
	replacedUser, err := h.usersController.ReplaceUser(r.Context(), id, user)
	if err != nil {
		if errors.Is(err, interfaces.ErrUserNotFound) {
			return scim.Resource{}, scimErrors.ScimErrorResourceNotFound(id)
		} else {
			return scim.Resource{}, scimErrors.ScimErrorInternal
		}
	}
	res, err := glue.UserToResource(replacedUser)
	if err != nil {
		return scim.Resource{}, scimErrors.ScimErrorInternal
	}
	return res, err
}
