# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import datetime
from functools import partial
import logging
from random import SystemRandom

from flask import (
    Flask,
    request,
)
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.social.api.common import (
    internal_error,
    not_found,
)
from passport.backend.social.api.logging_settings import logging_settings_init
from passport.backend.social.api.project import PROJECT_NAME
from passport.backend.social.api.views import v3 as views_v3
from passport.backend.social.api.views.v2 import (
    tasks as tasks_v2,
    views as views_v2,
)
from passport.backend.social.common import middleware
from passport.backend.social.common.db.utils import (
    build_master_db_config,
    build_slave_db_config,
    create_engine,
)
from passport.backend.social.common.grants import GrantsConfig
from passport.backend.social.common.limits import QLimits
from passport.backend.social.common.misc import (
    GraphiteMessageType,
    write_graph_log_message,
)
from passport.backend.social.common.provider_settings import providers as provider_settings
from passport.backend.social.common.redis_client import RedisClient
from passport.backend.social.common.services_settings import services as services_settings
from passport.backend.social.common.social_config import social_config
from passport.backend.social.common.tvm import SocialTvmCredentialsManager
from passport.backend.social.common.useragent import (
    build_http_pool_manager,
    ZoraUseragent,
)
from passport.backend.social.common.web_service import (
    Request,
    Response,
)
import passport.backend.social.proxylib
import werkzeug.exceptions


logger = logging.getLogger(__name__)


def create_app():
    app = Flask(__name__)
    app.request_class = Request
    app.response_class = Response

    def add_rules_for_api_version1(app):
        """
        Маппинг роутинга api в api2
        """
        app.add_url_rule(
            '/api/ping', methods=['GET'],
            view_func=views_v2.ping_view)

        app.add_url_rule(
            '/api/profile/<int:profile_id>', methods=['GET'],
            view_func=views_v2.get_profile)  # данные профиля
        app.add_url_rule(
            '/api/user/<int:uid>/profile', methods=['GET'],
            view_func=views_v2.get_profiles)  # список профилей
        app.add_url_rule(
            '/api/profile/<int:profile_id>', methods=['POST'],
            view_func=views_v2.edit_profile)  # изменение профиля
        app.add_url_rule(
            '/api/profile/<int:profile_id>', methods=['DELETE'],
            view_func=views_v2.delete_profile)  # удаление профиля
        app.add_url_rule(
            '/api/user/<int:uid>', methods=['DELETE'],
            view_func=views_v2.delete_user)  # удаление пользователя

        app.add_url_rule(
            '/api/token/<int:token_id>', methods=['GET'],
            view_func=views_v2.get_token)  # данные токена
        app.add_url_rule(
            '/api/profile/<int:profile_id>/token', methods=['POST'],
            view_func=views_v2.create_token)  # создание токена
        app.add_url_rule(
            '/api/token/<int:token_id>', methods=['POST'],
            view_func=views_v2.edit_token)  # изменение токена
        app.add_url_rule(
            '/api/token/<int:token_id>', methods=['DELETE'],
            view_func=views_v2.delete_token)  # удаление токена

    app.add_url_rule(
        '/api/profile', methods=['PUT'],
        view_func=views_v2.edit_profile)
    app.add_url_rule(
        "/api/profile", methods=['DELETE'],
        view_func=views_v2.delete_profile)

    app.add_url_rule(
        '/api/profile/<int:profile_id>/subscription/<int:sid>',
        methods=['PUT'],
        view_func=views_v2.create_subscription)
    app.add_url_rule(
        '/api/profile/<int:profile_id>/subscription/<int:sid>',
        methods=['DELETE'],
        view_func=views_v2.delete_subscription)

    app.add_url_rule(
        '/api/profiles', methods=['GET'],
        view_func=views_v2.get_profiles)

    app.add_url_rule(
        '/api/profiles/search', methods=['GET'],
        view_func=views_v2.search_profiles)

    app.add_url_rule(
        '/api/token', methods=['GET'],
        view_func=views_v2.get_token)
    app.add_url_rule(
        '/api/token', methods=['POST'],
        view_func=views_v2.create_token)
    app.add_url_rule(
        '/api/token', methods=['PUT'],
        view_func=views_v2.edit_token)
    app.add_url_rule(
        '/api/token', methods=['DELETE'],
        view_func=views_v2.delete_token)

    app.add_url_rule(
        '/api/task/<task_id>', methods=['GET'],
        view_func=tasks_v2.get_task)
    app.add_url_rule(
        '/api/task/<task_id>/bind', methods=['POST'],
        view_func=tasks_v2.create_profile_task)
    app.add_url_rule(
        '/api/task/<task_id>', methods=['DELETE'],
        view_func=tasks_v2.delete_task)

    app.add_url_rule(
        '/api/token/newest', methods=['GET'],
        view_func=views_v2.get_token_newest)
    app.add_url_rule(
        '/api/token/available', methods=['GET', 'POST'],
        view_func=views_v2.is_token_available)
    app.add_url_rule(
        '/api/token/delete_from_account', methods=['POST'],
        view_func=views_v2.DeleteTokensFromAccount.as_view())

    app.add_url_rule(
        '/api/special/who_shares_payment_cards', methods=['GET'],
        view_func=views_v3.WhoSharesPaymentCards.as_view())
    app.add_url_rule(
        '/api/special/who_shares_taxi_data', methods=['GET'],
        view_func=views_v3.WhoSharesTaxiData.as_view())
    app.add_url_rule(
        '/api/special/who_shares_taxi_data_v2', methods=['GET'],
        view_func=views_v3.WhoSharesTaxiDataV2.as_view())

    app.add_url_rule(
        '/api/delete_social_data', methods=['POST'],
        view_func=views_v3.DeleteSocialData.as_view())
    app.add_url_rule(
        '/api/unbind_phonish_by_uid', methods=['POST'],
        view_func=views_v3.UnbindPhonishByUid.as_view())

    app.add_url_rule(
        '/api/get_station_applications', methods=['GET', 'POST'],
        view_func=views_v3.GetStationApplications.as_view())
    app.add_url_rule(
        '/api/create_station_application', methods=['POST'],
        view_func=views_v3.CreateStationApplication.as_view())
    app.add_url_rule(
        '/api/remove_station_application', methods=['POST'],
        view_func=views_v3.RemoveStationApplication.as_view())
    app.add_url_rule(
        '/api/change_station_application', methods=['POST'],
        view_func=views_v3.ChangeStationApplication.as_view())

    add_rules_for_api_version1(app)

    app.before_request(middleware.setup_request)
    app.before_request(middleware.log_http_request)
    app.before_request(middleware.log_api_call)
    app.after_request(partial(middleware.after_request, 'social-api-access-log'))
    app.errorhandler(Exception)(handle_exception)
    app.errorhandler(404)(handle_page_not_found)
    app.errorhandler(werkzeug.exceptions.MethodNotAllowed)(handle_page_not_found)
    return app


def prepare_interprocess_environment():
    """
    Инициализация подсистем, которые могут быть общими для нескольких процессов.

    Здесь следует инициализировать только подсистемы, которые не меняют своего
    состояния на протяжении жизни приложения.
    """
    logging_settings_init()

    social_config.init()

    LazyLoader.register('qlimits', QLimits)
    LazyLoader.get_instance('qlimits')

    provider_settings.init()
    services_settings.init()
    passport.backend.social.proxylib.init()


def prepare_intraprocess_environment():
    """
    Инициализация подсистем, которые нельзя делать общими для нескольких
    процессов.

    Здесь следует инициализировать подсистемы, которые меняют своё состояние
    на протяжении жизни приложения.
    """
    LazyLoader.register('chrono', lambda: datetime.datetime)
    LazyLoader.get_instance('chrono')

    LazyLoader.register('randomizer', SystemRandom)
    LazyLoader.get_instance('randomizer')

    LazyLoader.register('http_pool_manager', build_http_pool_manager)
    LazyLoader.get_instance('http_pool_manager')

    read_conn = create_engine(build_slave_db_config(social_config), False)
    LazyLoader.register('slave_db_engine', lambda: read_conn)
    LazyLoader.get_instance('slave_db_engine')

    write_conn = create_engine(build_master_db_config(social_config), False)
    LazyLoader.register('master_db_engine', lambda: write_conn)
    LazyLoader.get_instance('master_db_engine')

    RedisClient.init()
    LazyLoader.register('redis', RedisClient)
    LazyLoader.get_instance('redis')

    tvm_credentials_manager = SocialTvmCredentialsManager()
    LazyLoader.register('TvmCredentialsManager', lambda: tvm_credentials_manager)
    LazyLoader.get_instance('TvmCredentialsManager')

    grants_config = GrantsConfig(
        PROJECT_NAME,
        tvm_credentials_manager=tvm_credentials_manager,
    )
    LazyLoader.register('grants_config', lambda: grants_config)
    LazyLoader.get_instance('grants_config')

    zora_useragent = ZoraUseragent(tvm_credentials_manager=tvm_credentials_manager)
    LazyLoader.register('zora_useragent', lambda: zora_useragent)
    LazyLoader.get_instance('zora_useragent')


def handle_page_not_found(_):
    return not_found('page-not-found')


def handle_exception(_):
    logger.error('Unhandled exception', exc_info=True)
    write_graph_log_message(GraphiteMessageType.error, request.endpoint, None, 'internal')
    return internal_error('Unhandled exception')
