# -*- coding: utf-8 -*-

import logging
from logging.config import dictConfig as logging_dict_config
import os
import sys

from passport.backend.social.common.context import request_ctx
from passport.backend.social.common.social_logging import (
    close_logging_handlers,
    ExceptionFormatter,
    SocialFormatter,
    WarningFormatter,
)


def logging_settings_init():
    logging.captureWarnings(True)
    conf = _get_logging_configuration()
    logging_dict_config(conf)


def logging_settings_deinit():
    conf = _get_logging_configuration()
    close_logging_handlers(conf)


def _get_logging_configuration():
    root_handlers = {'api', 'api.error', 'api.exception', 'api.warning'}
    subprocess_handlers = {'api.subprocess.error'}
    if not os.environ.get('SYSTEM_WIDE_SOCIAL_API'):
        root_handlers.add('console')
        subprocess_handlers.add('console')

    if os.environ.get('SYSTEM_WIDE_SOCIAL_API'):
        LOG_PATH = '/var/log/yandex/socialism'
    else:
        LOG_PATH = '.'

    return {
        'version': 1,
        'disable_existing_loggers': False,
        'formatters': {
            'default': {
                '()': SocialFormatter,
                'context': request_ctx,
            },
            'exception': {
                '()': ExceptionFormatter,
                'context': request_ctx,
                'logtype': 'social-api-exception-log',
            },
            'warning': {
                '()': WarningFormatter,
                'context': request_ctx,
                'logtype': 'social-api-warning-log',
            },
            'simple': {
                'format': "%(message)s"
            },
        },
        'root': {
            'handlers': root_handlers,
            'level': 'DEBUG',
        },
        'handlers': {
            'console': {
                'class': 'logging.StreamHandler',
                'stream': sys.stdout,
                'level': 'DEBUG',
                'formatter': 'default',
            },
            'api': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': os.path.join(LOG_PATH, 'social-api.debug.log'),
                'formatter': 'default',
                'level': 'DEBUG',
            },
            'api.error': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': os.path.join(LOG_PATH, 'social-api.error.log'),
                'formatter': 'default',
                'level': 'ERROR',
            },
            'api.exception': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': os.path.join(LOG_PATH, 'social-api.exception.log'),
                'formatter': 'exception',
                'level': 'ERROR',
                'filters': ['exception_filter'],
            },
            'api.warning': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': os.path.join(LOG_PATH, 'social-api.warning.log'),
                'formatter': 'warning',
                'filters': ['warning_filter'],
            },
            'api.subprocess.error': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': os.path.join(LOG_PATH, 'social-api.subprocess.error.log'),
                'formatter': 'default',
                'level': 'DEBUG',
            },
            'bindings_log': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': os.path.join(LOG_PATH, 'social-bindings.statbox.log'),
                'formatter': 'simple',
            },
            'statbox': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': os.path.join(LOG_PATH, 'social-api.statbox.log'),
                'formatter': 'simple',
            },
            'access': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': os.path.join(LOG_PATH, 'social-api.access.log'),
                'formatter': 'simple',
            },
            'graphite': {
                'class': 'passport.backend.social.common.social_logging.OsFileHandler',
                'filename': os.path.join(LOG_PATH, 'social-api.graphite.log'),
                'formatter': 'simple',
            },
        },
        'filters': {
            'exception_filter': {
                '()': 'passport.backend.social.common.social_logging.ExceptionFilter',
            },
            'warning_filter': {
                '()': 'passport.backend.social.common.social_logging.LevelFilter',
                'levels': ['WARNING'],
            },
        },
        'loggers': {
            'subprocess.error': {
                'handlers': subprocess_handlers,
                'propagate': False,
            },
            'urllib3': {
                'level': logging.INFO,
            },
            'bindings_log': {
                'handlers': ['bindings_log'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'statbox': {
                'handlers': ['statbox'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'access': {
                'handlers': ['access'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'graphite.database': {
                'handlers': ['graphite'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'graphite.useragent': {
                'handlers': ['graphite'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'graphite.redis': {
                'handlers': ['graphite'],
                'level': 'DEBUG',
                'propagate': False,
            },
            'passport.backend.social.common._urllib3': {'level': 'WARNING'},
            'passport.backend.social.common.db.execute': {'level': 'WARNING'},
            'passport.backend.social.common.useragent': {'level': 'INFO'},

            # блок конфигурации для билдеров
            'passport.blackbox': {
                'handlers': root_handlers - {'api.error'},
                'level': 'DEBUG',
                'propagate': False,
            },
            'passport.backend.social.common.builders.billing': {
                'handlers': root_handlers - {'api.error'},
                'level': 'DEBUG',
                'propagate': False,
            },
        },
    }
